/**
  *  \file sigslot.h
  *  \brief Signal / Slot
  */
#ifndef STREU_CPLUSLIB_SIGSLOT_H_INCLUDED
#define STREU_CPLUSLIB_SIGSLOT_H_INCLUDED

/** Base class for a signal handler. Together with SignalConnection,
    this provides lifetime management for signal connections. Users
    never directly interact with this class, except by passing the
    SignalHandler* they got from SignalX.add() to
    SignalConnection(). */
class SignalHandler {
    friend class SignalConnection;
    void operator=(const SignalHandler&);
    SignalHandler(const SignalHandler&);

    /** Points to the pointer to this SignalHandler in a
        SignalConnection. May be null. */
    SignalHandler** backlink;
    /** Pointer to previous element's next link in signal chain. */
    SignalHandler** prev;
 public:
    /** Next signal handler in this signal chain. */
    SignalHandler*  next;
 protected:
    /** Constructor. Sets up pointers for hooking this element after
        *prev, but does not modify *prev yet. */
    SignalHandler(SignalHandler** prev)
        : backlink(0), prev(prev), next(*prev)
        {
            if (next)
                next->prev = &next;
        }
 public:
    virtual ~SignalHandler();
};

/** Signal connection. This provides lifetime management for signal
    connections.

    A signal receiver initialize this object with the result of some
    sig.add(...). This will ensure that the signal is disconnected
    when the receiver dies. It's also possible that the sender dies,
    in this case the SignalConnection will automatically revert to
    unconnected state. */
class SignalConnection {
    SignalHandler* handler;

    void operator=(const SignalConnection&);
    SignalConnection(const SignalConnection&);
 public:
    SignalConnection(SignalHandler* handler);
    /** Constructor. Creates a disconnected SignalConnection. */
    SignalConnection()
        : handler(0)
        { }
    ~SignalConnection()
        { delete handler; }
    /** Disconnect this signal. */
    void disconnect()
        { delete handler; }
    void operator=(SignalHandler* handler);
    /** Check whether this connection is occupied. */
    bool isConnected() const
        { return handler != 0; }
};

/** Zero-argument Signal. Accepts only parameterless signal
    handlers, in objects or static functions. */
class Signal0 {
    /// Basic handler.
    struct Handler : public SignalHandler {
        Handler(SignalHandler** prev) : SignalHandler(prev) { }
        virtual void raise() = 0;
    };

    /// Bound member function handler.
    template<class A>
    struct ObjHandler : Handler {
        A* obj;
        void (A::*pmf)();

        ObjHandler(SignalHandler** prev, A* obj, void (A::*pmf)())
            : Handler(prev), obj(obj), pmf(pmf) { }
        void raise() { (obj->*pmf)(); }
    };

    /// Static function handler.
    struct StaticHandler : Handler {
        void (*func)();

        StaticHandler(SignalHandler** prev, void (*func)())
            : Handler(prev), func(func) { }
        void raise() { func(); }
    };

    /// Signal handler chain.
    SignalHandler* list;
 public:
    /** Constructor. */
    Signal0()
        : list(0) { }

    /** Add static function. */
    SignalHandler* add(void (*func)()) {
        return list = new StaticHandler(&list, func);
    }

    /** Add bound member function. */
    template<class A>
    SignalHandler* add(A* ptr, void (A::*pmf)()) {
        return list = new ObjHandler<A>(&list, ptr, pmf);
    }

    /** Raise signal. Calls all registered handlers. The handler
        registered last will be called first. */
    void raise();

    /** Destructor. */
    ~Signal0();
};

#define DECLARE_SIGNAL(TEMPLATE, NAME, DECL, NNDECL, CALL)      \
/** Parameterized signal. This signal is raised with            \
    parameters which are passed to all handlers. It accepts     \
    handlers with parameters (and correct signature) as well    \
    as parameterless handlers. */                               \
TEMPLATE                                                        \
class NAME {                                                    \
    /** Base class for signal handler */                        \
    struct Handler : public SignalHandler {                     \
        Handler(SignalHandler** prev) : SignalHandler(prev) { } \
        virtual void raise(DECL) = 0;                           \
    };                                                          \
                                                                \
    /** Bound member function handler. */                       \
    template<class A>                                           \
    struct ObjHandler : public Handler {                        \
        A* obj;                                                 \
        void (A::*method)(DECL);                                \
        ObjHandler(SignalHandler** prev, A* aobj, void (A::*amethod)(DECL)) \
            : Handler(prev), obj(aobj), method(amethod) { }     \
        void raise(DECL) { (obj->*method)(CALL); }              \
    };                                                          \
                                                                \
    /** Parameterless bound member function handler. */         \
    template<class A>                                           \
    struct ObjHandlerNA : public Handler {                      \
        A* obj;                                                 \
        void (A::*method)();                                    \
        ObjHandlerNA(SignalHandler** prev, A* aobj, void (A::*amethod)()) \
            : Handler(prev), obj(aobj), method(amethod) { }     \
        void raise(NNDECL) { (obj->*method)(); }                \
    };                                                          \
                                                                \
    /** Static function handler. */                             \
    struct StaticHandler : public Handler {                     \
        void (*func)(DECL);                                     \
        StaticHandler(SignalHandler** prev, void (*afunc)(DECL)) \
            : Handler(prev), func(afunc) { }                    \
        void raise(DECL) { func(CALL); }                        \
    };                                                          \
                                                                \
    /** Static parameterless function handler. */               \
    struct StaticHandlerNA : public Handler {                   \
        void (*func)();                                         \
        StaticHandlerNA(SignalHandler** prev, void (*afunc)())  \
            : Handler(prev), func(afunc) { }                    \
        void raise(NNDECL) { func(); }                          \
    };                                                          \
                                                                \
    /** Handler list */                                         \
    SignalHandler* list;                                        \
 public:                                                        \
                                                                \
    /** Constructor. */                                         \
    NAME() : list(0) { }                                        \
                                                                \
    /** Destructor. */                                          \
    ~NAME();                                                    \
                                                                \
    /** Add bound member function handler.                      \
        sig.add(foo, &Foo::::method)                            \
        -> call foo->method(x) when signal is raised            \
        with raise(x). */                                       \
    template<class A>                                           \
    SignalHandler* add(A* obj, void (A::*method)(DECL)) {       \
        return list = new ObjHandler<A>(&list, obj, method);    \
    }                                                           \
                                                                \
    /** Add bound member function handler.                      \
        sig.add(foo, &Foo::::method)                            \
        -> call foo->method() when signal is raised. */         \
    template<class A>                                           \
    SignalHandler* add(A* obj, void (A::*method)()) {           \
        return list = new ObjHandlerNA<A>(&list, obj, method);  \
    }                                                           \
                                                                \
    /** Add static function handler.                            \
        sig.add(function)                                       \
        -> call function(x) when signal is raised               \
        with raise(x). */                                       \
    SignalHandler* add(void (*func)(DECL)) {                    \
        return list = new StaticHandler(&list, func);           \
    }                                                           \
                                                                \
    /** Add static function handler.                            \
        sig.add(function)                                       \
        -> call function() when signal is raised */             \
    SignalHandler* add(void (*func)()) {                        \
        return list = new StaticHandlerNA(&list, func);         \
    }                                                           \
                                                                \
    /** Raise signal with specified parameter(s).               \
        Calls all handlers. The handler added last will         \
        be called first. */                                     \
    void raise(DECL);                                           \
};

#define DEFINE_SIGNAL(TEMPLATE, NAME, NNN, DECL, NNDECL, CALL)          \
TEMPLATE                                                                \
void                                                                    \
NAME::raise(DECL)                                                       \
{                                                                       \
    SignalHandler* q = list;                                            \
    while(q) {                                                          \
        /* remember the 'here' in case the handler removes              \
           itself. FIXME: make sure that 'q' isn't removed              \
           either. */                                                   \
        Handler* here = static_cast<Handler*>(q);                       \
        q = q->next;                                                    \
        here->raise(CALL);                                              \
    }                                                                   \
}                                                                       \
                                                                        \
TEMPLATE                                                                \
NAME::~NNN()                                                            \
{                                                                       \
    while (list) {                                                      \
       delete list;                                                     \
    }                                                                   \
}

#define AND ,
DECLARE_SIGNAL (template<class T>, Signal1,    T arg, T, arg);
DEFINE_SIGNAL  (template<class T>, Signal1<T>, Signal1, T arg, T, arg);
DECLARE_SIGNAL (template<class T AND class T2>, Signal2,           T arg AND T2 arg2, T AND T2, arg AND arg2);
DEFINE_SIGNAL  (template<class T AND class T2>, Signal2<T AND T2>, Signal2, T arg AND T2 arg2, T AND T2, arg AND arg2);
DECLARE_SIGNAL (template<class T AND class T2 AND class T3>, Signal3,                           T arg AND T2 arg2 AND T3 arg3, T AND T2 AND T3, arg AND arg2 AND arg3);
DEFINE_SIGNAL  (template<class T AND class T2 AND class T3>, Signal3<T AND T2 AND T3>, Signal3, T arg AND T2 arg2 AND T3 arg3, T AND T2 AND T3, arg AND arg2 AND arg3);
DECLARE_SIGNAL (template<class T AND class T2 AND class T3 AND class T4>, Signal4,                                  T arg AND T2 arg2 AND T3 arg3 AND T4 arg4, T AND T2 AND T3 AND T4, arg AND arg2 AND arg3 AND arg4);
DEFINE_SIGNAL  (template<class T AND class T2 AND class T3 AND class T4>, Signal4<T AND T2 AND T3 AND T4>, Signal4, T arg AND T2 arg2 AND T3 arg3 AND T4 arg4, T AND T2 AND T3 AND T4, arg AND arg2 AND arg3 AND arg4);
#undef DECLARE_SIGNAL
#ifndef STREU_CPLUSLIB_KEEP_DEFINE_SIGNAL
#  undef DEFINE_SIGNAL
#endif
#undef AND

/** Observable object. This provides a simple signal-if-changed
    mechanism. The Observable contains an object of type T, and raises
    sig_changed whenever the value changes. Assignments of the current
    value, as in o.set(o.get()), do not raise the signal. */
template<class T>
class Observable {
    T value;
 public:
    /** Constructor. */
    Observable(T value)
        : value(value) { }
    ~Observable();

    /** Set new value. */
    void set(T newv);

    /** Get current value. */
    T get() const
        { return value; }

    /** Change signal. Raised if value changes. */
    Signal1<T> sig_changed;
};

template<class T>
Observable<T>::~Observable()
{ }

template<class T>
void
Observable<T>::set(T newv)
{
    if (newv != value) {
        value = newv;
        sig_changed.raise(value);
    }
}

#endif
