/**
  *  \file cpluslib/object.h
  *  \brief Base Class
  *  \author Stefan Reuther
  */
#ifndef STREU_CPLUSLIB_OBJECT_H_INCLUDED
#define STREU_CPLUSLIB_OBJECT_H_INCLUDED

#include "cpluslib/config.h"
#if defined(STREU_CONFIG_OBJECT_HAS_TOSTRING)
#  include "cpluslib/string.h"
#endif

class Object;

/** Reference counter for smart pointer. Implemented in smartptr.cc. */
struct ReferenceCount {
    std::size_t ptr_count;      ///< Number of Ptrs referring here.

    /// Get reference count for array.
    static ReferenceCount* getRefCountForArray(void* ptr)
        { return getRefCountForPointer(ptr); }
    /// Get reference count for pointer.
    static ReferenceCount* getRefCountForPointer(void* ptr);
#ifdef STREU_CONFIG_PTR_OPTIMIZE_OBJECT
    /// Get reference count for Object.
    static inline ReferenceCount* getRefCountForPointer(Object* ptr);
#endif
    /// Null reference count. Used for null pointers.
    static ReferenceCount null_ref;
};

/*! \class Object
    \brief Generic "complicated" object

    This object is intended as a root for a complex class hierarchy. 
    It provides
    - integration with Ptr<> (optional, STREU_CONFIG_PTR_OPTIMIZE_OBJECT)
    - a virtual destructor (which is a must for hierarchies)
    - a toString() method which is useful in many cases (optional,
      STREU_CONFIG_OBJECT_HAS_TOSTRING)
    - ObjectHolder can only store Object descendants */
#ifdef STREU_CONFIG_PTR_OPTIMIZE_OBJECT
// Implementation with reference count.
class Object {
 private:
    ReferenceCount rc;
    friend ReferenceCount* ReferenceCount::getRefCountForPointer(Object*);
 public:
    Object()
        { rc.ptr_count = 0; }
    Object(const Object& rhs)
        { rc.ptr_count = 0; }
    Object& operator=(const Object& rhs)
        { return *this; }
    virtual ~Object();
#ifdef STREU_CONFIG_OBJECT_HAS_TOSTRING
    virtual string_t toString();
#endif
};

inline ReferenceCount*
ReferenceCount::getRefCountForPointer(Object* ptr)
{
    return ptr ? &ptr->rc : &null_ref;
}
#else
// Implementation without reference count.
class Object {
 public:
    virtual ~Object();
#ifdef STREU_CONFIG_OBJECT_HAS_TOSTRING
    virtual string_t toString();
#endif
};
#endif

#endif
