/**
  *  \file maybe.h
  *  \brief Maybe types
  *  \author Stefan Reuther
  *
  *  This contains classes to implement "Maybe" values, that is,
  *  objects which may or may not contain a value.
  */
#ifndef PCC_V2_MAYBE_H
#define PCC_V2_MAYBE_H

#include "cpluslib/assert.h"

/** Simple Maybe class. This class provides a simple way to pass
    around values which might mean 'unknown'. It does not enforce any
    restrictions, though. For example, multiplication of two 'Maybe'
    values which happen to be unknown will not necessarily yield an
    unknown value again. It will just multiply their representations. 
    It is your responsibility to check in advance. On the plus side,
    this yields very efficient code. The idea is not to enforce
    anything, but just document whenever such 'maybe-unknown' values
    are passed around and what actually means 'unknown' at this
    particular place.

    For debugging, this contains the appropriate checks, which prevent
    you from accidentially using a null value.

    The class is intentionally minimalistic so conversions between
    different SimpleMaybe instanciations require casts.

    \param N type
    \param null value used to mean 'unknown' */
template<typename N, N null>
class SimpleMaybe {
    N value;

    /** Conversion to other maybe types is not allowed. It must be
        made explicit using getValue().  */
    template<typename N1, N1 null1>
    SimpleMaybe(SimpleMaybe<N1,null1>);
 public:
    /** Constructor. Create a copy of value (which may be the null
        value or anything else). */
    explicit SimpleMaybe(N value)
        : value(value) { }
    /** Constructor. Create a null (unknown) value. */
    SimpleMaybe()
        : value(null) { }
    /** Check whether this value is known. */
    bool isKnown() const
        { return value != null; }
    /** Convert to base type. This operator is used when the maybe
        object participates in an arithmetic expression, for
        example.
        \pre isValid() */
    operator N() const
        { return getValue(); }
    /** Convert to base type.
        \pre isValid() */
    N getValue() const
        {
            ASSERT(isKnown());
            return value;
        }
    /** Get raw value. */
    N getRawValue() const
        { return value; }
    /** Compare for equality. */
    bool operator==(SimpleMaybe<N,null> other) const
        { return value == other.value; }
    /** Compare for inequality. */
    bool operator!=(SimpleMaybe<N,null> other) const
        { return value != other.value; }
};

/** Explicit Maybe class. This class provides a simple way to pass
    around possibly-unknown values. Like SimpleMaybe, this one does
    not enforce any "tristate" properties.

    The class is intentionally minimalistic so conversions between
    different Maybe instanciations require casts.

    \todo optimize this by doing explicit construction and destruction
    to not carry around an object when we don't need it

    \param N type */
template<typename N>
class Maybe {
    bool valid;
    N value;

    /** Conversion to other maybe types is not allowed. It must be
        made explicit.  */
    template<typename N1>
    Maybe(Maybe<N1>);
 public:
    /** Construct known value.
        \param N object to copy */
    explicit Maybe(const N& value)
        : valid(true), value(value) { }
    /** Construct unknown value. */
    Maybe()
        : valid(false) { }
    /** Check whether this value is known. */
    bool isKnown() const
        { return valid; }
    /** Convert to base type. This operator is used when the maybe
        object participates in an arithmetic expression, for
        example.
        \pre isValid() */
    operator const N&() const
        { return getValue(); }
    /** Convert to base type.
        \pre isValid() */
    const N& getValue() const
        {
            ASSERT(isKnown());
            return value;
        }
    /** Compare for inequality. Unknown values compare equal. */
    bool operator==(const Maybe<N>& other) const
        {
            return (valid
                    ? (other.valid && value == other.value)
                    : !other.valid);
        }
    bool operator!=(const Maybe<N>& other) const
        { return !(operator==(other)); }
};

#endif
