/**
  *  \file let.h
  *  \brief Quasi-Dynamic Scope
  */
#ifndef STREU_CPLUSLIB_LET_H_INCLUDED
#define STREU_CPLUSLIB_LET_H_INCLUDED

/** Restorer.
    Saves the value of a variable. When the restorer goes out of scope,
    the value is restored.
    \code
      int v = 19;
      {
          restorer<int> int_restorer(v);
          v = 42;
      }
      // at this place, v is again 19
    \endcode */
template<class T>
class restorer {
    void operator=(const restorer&);
    restorer(const restorer&);
    T old_value;
    T* ref;
 public:
    /** Semi-Copy constructor. When the compiler elides constructors,
        this one is never used. g++ seems not to do this, but at least
        optimizes away the test in the dtor. */
    restorer (restorer& other)
        : old_value(other.old_value), ref(other.ref)
        { other.ref = 0; }
    restorer (T& value)
        : old_value(value), ref(&value) { }
    ~restorer()
        {
            if (ref)
                *ref = old_value;
        }
};

/** Quasi-Dynamic Scope.
    Use as
    \code
       let (var, newvalue),
           foo();
    \endcode
    %foo() will see the value newvalue in the specified variable
    (which should be a member or global variable). */
template<class T, class U>
restorer<T>
let(T& which, const U& value)
{
    restorer<T> v(which);
    which = value;
    return v;
}

#endif
