/**
  *  \file format.h
  *  \brief Typesafe "sprintf" workalike
  *  \author Stefan Reuther
  */
#ifndef STREU_CPLUSLIB_FORMAT_H_INCLUDED
#define STREU_CPLUSLIB_FORMAT_H_INCLUDED

#include "cpluslib/string.h"
#include <iosfwd>

class format;

string_t toString(char c, char, int, int, int, format&);
string_t toString(unsigned char c, char, int, int, int, format&);
string_t toString(unsigned short c, char, int, int, int, format&);
string_t toString(unsigned int c, char, int, int, int, format&);
string_t toString(unsigned long c, char, int, int, int, format&);
string_t toString(signed char c, char, int, int, int, format&);
string_t toString(signed short c, char, int, int, int, format&);
string_t toString(signed int c, char, int, int, int, format&);
string_t toString(signed long c, char, int, int, int, format&);
string_t toString(float c, char, int, int, int, format&);
string_t toString(double c, char, int, int, int, format&);
string_t toString(long double c, char, int, int, int, format&);
string_t toString(const char* c, char, int, int, int, format&);
string_t toString(const string_t& s, char, int, int, int, format&);
string_t toString(const void* c, char, int, int, int, format&);

/** Argument storage for format. This union contains one argument for
    the formatting. It can either contain a pointer to a (temporary)
    object, or an unsigned long. */
union format_union {
    const void*   cv;
    unsigned long ul;
};

/** Format Traits. This defines how to remember an object in a
    format_union. */
template<class T>
struct format_traits {
    static void store(format_union& u, const T& t)
        { u.cv = &t; }
    static const T& load(const format_union& u)
        { return *static_cast<const T*>(u.cv); }
};

#define FORMAT_TRAITS_SPECIALIZE(TYPE)                          \
template<>                                                      \
struct format_traits<TYPE> {                                    \
    static void store(format_union& u, TYPE c) { u.ul = c; }    \
    static TYPE load(const format_union& u) { return u.ul; }    \
}
FORMAT_TRAITS_SPECIALIZE(char);
FORMAT_TRAITS_SPECIALIZE(signed char);
FORMAT_TRAITS_SPECIALIZE(unsigned char);
FORMAT_TRAITS_SPECIALIZE(short);
FORMAT_TRAITS_SPECIALIZE(unsigned short);
FORMAT_TRAITS_SPECIALIZE(int);
FORMAT_TRAITS_SPECIALIZE(unsigned int);
FORMAT_TRAITS_SPECIALIZE(long);
FORMAT_TRAITS_SPECIALIZE(unsigned long);

template<class T>
string_t printHelper(format_union u, char typ, int wi, int pre, int flags, format& fmt) {
    return toString(format_traits<T>::load(u), typ, wi, pre, flags, fmt);
}

/** Typesafe "sprintf" workalike.
    See documentation of format.cc for more details.

    Why `class format'? This allows us to do
    `format("...") << ....' without having to copy these beasts.
    Making `format' a function does not guarantee that. */
class format {
    struct ArgClosure {
        format_union data;
        string_t (*func)(format_union, char, int, int, int, format&);
    };
 protected:
    enum {
        MAX_ARGS = 10
    };
    const char* fmt;
    ArgClosure fac[MAX_ARGS];
    int n;
 public:
    unsigned int state;

    enum {
        /* These are bit values */
        FORMAT_ALTERNATE = 1,   /* # alternate format */
        FORMAT_ZEROPAD   = 2,   /* 0 zero pad, overrides '-' */
        FORMAT_LEFTJUST  = 4,   /* - left justify */
        FORMAT_BLANK     = 8,   /*   prepend blank when positive */
        FORMAT_SIGN      = 16,  /* + show positive sign if nonnegative, overrides ' ' */
        FORMAT_GROUP     = 32   /* ' use locale-specific grouping. not usable with '0' */
    };
    enum {
        /* These are bit numbers */
        STATE_NULL = 0,         /* value was null or empty */
        STATE_ONE  = 1          /* value was one */
    };

    /** Construct formatter object. */
    format(const char* afmt)
        : fmt(afmt), n(0) { }

    /** Construct formatter object with one parameter. */
    template<class A>
    format(const char* afmt, const A& arg)
        : fmt(afmt), n(1)
        {
            format_traits<A>::store(fac[0].data, arg);
            fac[0].func = &printHelper<A>;
        }

    /** Construct formatter object with two parameters. */
    template<class A, class B>
    format(const char* afmt, const A& arg, const B& brg)
        : fmt(afmt), n(2)
        {
            format_traits<A>::store(fac[0].data, arg);
            fac[0].func = &printHelper<A>;
            format_traits<B>::store(fac[1].data, brg);
            fac[1].func = &printHelper<B>;
        }

    /** Construct formatter object with three parameters. */
    template<class A, class B, class C>
    format(const char* afmt, const A& arg, const B& brg, const C& crg)
        : fmt(afmt), n(3)
        {
            format_traits<A>::store(fac[0].data, arg);
            fac[0].func = &printHelper<A>;
            format_traits<B>::store(fac[1].data, brg);
            fac[1].func = &printHelper<B>;
            format_traits<C>::store(fac[2].data, crg);
            fac[2].func = &printHelper<C>;
        }

    /** Construct formatter object with four parameters. */
    template<class A, class B, class C, class D>
    format(const char* afmt, const A& arg, const B& brg,
           const C& crg, const D& drg)
        : fmt(afmt), n(4)
        {
            format_traits<A>::store(fac[0].data, arg);
            fac[0].func = &printHelper<A>;
            format_traits<B>::store(fac[1].data, brg);
            fac[1].func = &printHelper<B>;
            format_traits<C>::store(fac[2].data, crg);
            fac[2].func = &printHelper<C>;
            format_traits<D>::store(fac[3].data, drg);
            fac[3].func = &printHelper<D>;
        }

    /** Add one parameter. */
    template<typename T>
    format& operator<<(const T& arg) {
        format_traits<T>::store(fac[n].data, arg);
        fac[n].func = &printHelper<T>;
        ++n;
        return *this;
    }

    /** Convert to string. This does the actual conversion. */
    operator string_t();

    /** Set status flag. Called by formatter functions to set bits for
        the "%0{" construct. */
    void setState(int n)
        { state |= 1U << n; }
    /** Clear status flag. Called by formatter functions to set bits for
        the "%0{" construct. */
    void clearState(int n)
        { state &= ~(1U << n); }
    /** Test status flag. Used by the "%0{" construct. */
    bool hasState(int n)
        { return state & (1 << n); }
};

std::ostream& operator<<(std::ostream& os, format& fmt);

#endif
