/**
  *  \file fixed_vector.h
  *  \brief Fixed-size vector
  *  \author Stefan Reuther
  */
#ifndef FIXED_VECTOR_H_INCLUDED
#define FIXED_VECTOR_H_INCLUDED

#include <cstddef>
#include <new>
#include <iterator>
#include <algorithm>
#include "cpluslib/assert.h"
#include "cpluslib/fixed_container.h"

/****************************** fixed_vector *****************************/

/** Fixed-size vector. This class provides an interface similar to
    std::vector, but uses a fixed chunk of memory instead of growing
    dynamically.

    Because you must pass pointers to the memory to manage into the
    constructor, this class has no copy constructor. All other
    functions remain the same as for vector.

    Distinctive properties:
    - Objects can be inserted and deleted in amortized constant time
      at the end only
    - Provides random-access iterators
    - Constant memory-management overhead

    Exception safety: if a copy constructor throws an exception, the
    fixed_vector may be left in an intermediate state. For example,
    insert() first moves the elements at the end of the container to
    their new place, and then overwrites them with copies of the
    elements to be inserted. If during the latter operation a copy
    constructor throws, the insert() is not rolled back; you will be
    left with a container containing some elements twice.

    Overflow handling: most functions will prevent overflow
    completely, i.e. they will first check whether the operation would
    overflow (and throw a fixed_container_overrun exception if that is
    the case), and only proceed if they will definitely not overflow. 
    The only exception are assign() and the constructor, which take
    input iterators, and therefore do not know in advance how many
    elements there will be. They may leave you with a
    partially-processed range.
*/
template<typename T>
class fixed_vector {
    fixed_vector(const fixed_vector&);
 public:
    typedef T&             reference;
    typedef const T&       const_reference;
    typedef T*             iterator;
    typedef const T*       const_iterator;
    typedef std::size_t    size_type;
    typedef std::ptrdiff_t difference_type;
    typedef T              value_type;
    typedef T*             pointer;
    typedef const T*       const_pointer;
    typedef std::reverse_iterator<iterator> reverse_iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
 public:
    /** Size of a vector element. This is the amount of storage the
        user must provide for each object. For a fixed_vector<T>, this
        is guaranteed to be the same as sizeof(T). */
    static const std::size_t node_size = sizeof(T);

    /** Constructor. Creates an empty vector.
        \param data      [in] Data space. User must make sure it is aligned
                         correctly. It must contain num_total*sizeof(T) bytes.
                         It must not contain constructed objects.
        \param num_total [in] Number of elements this fixed_vector is expected
                         to hold. */
    explicit fixed_vector(void* data, size_type num_total)
        : data(static_cast<T*>(data)), num_total(num_total), num_valid(0)
        { }

    /** Constructor. Creates a vector containing \c n copies of \c value.
        \param data      [in] Data space. See fixed_vector(void*,size_type).
        \param num_total [in] Number of elements. See fixed_vector(void*,size_type).
        \param n         [in] Number of initial elements.
        \param value     [in] Value to put into vector initially.
        \pre n <= num_total */
    explicit fixed_vector(void* data, size_type num_total, size_type n, const T& value)
        : data(static_cast<T*>(data)), num_total(num_total), num_valid(0)
        {
            this->assign(n, value);
        }

    /** Constructor. Creates a vector containing a copy of the given range.
        \param data       [in] Data space. See fixed_vector(void*,size_type).
        \param num_total  [in] Number of elements. See fixed_vector(void*,size_type).
        \param first,last [in] Range of initial elements, input iterators. */
    template<typename It>
    fixed_vector(void* data, size_type num_total, It first, It last)
        : data(static_cast<T*>(data)), num_total(num_total), num_valid(0)
        {
            this->assign(first, last);
        }

    /** Constructor. Creates a vector containing a copy of another vector.
        \param data      [in] Data space. See fixed_vector(void*,size_type).
        \param num_total [in] Number of elements. See fixed_vector(void*,size_type).
        \param other     [in] Vector to copy. */
    fixed_vector(void* data, size_type num_total, const fixed_vector& other)
        : data(static_cast<T*>(data)), num_total(num_total), num_valid(0)
        {
            this->operator=(other);
        }

    /** Destructor. */
    virtual ~fixed_vector()
        { clear(); }

    /** Assignment operator. */
    fixed_vector& operator=(const fixed_vector& other);

    /** Assign range. Replaces the vector's contents with the contents
        of [first,last).
        \param first,last Range, input iterators */
    template<typename It>
    void assign(It first, It last);

    /** Assign value repeatedly. Replaces the vector's contents with
        \c n copies of \c value. */
    void assign(size_type n, const T& value);

    /** Begin iterator. \return Modifyable random access iterator to beginning. */
    iterator begin()
        { return data; }
    /** Begin iterator. \return Constant random access iterator to beginning. */
    const_iterator begin() const
        { return data; }

    /** End iterator. \return Modifyable random access iterator one-past-the-end. */
    iterator end()
        { return &data[num_valid]; }
    /** End iterator. \return Constant random access iterator one-past-the-end. */
    const_iterator end() const
        { return &data[num_valid]; }

    // rbegin, rend

    /** Get size of vector. \return current number of elements. */
    size_type size() const
        { return num_valid; }
    /** Get maximum size. \return maximum number of elements, as set in the constructor. */
    size_type max_size() const
        { return num_total; }
    /** Change size of vector. If \c new_size is less than size(),
        removes the excess elements. If \c new_size is larger than
        size(), enlarge the vector by appending copies of \c value. */
    void resize(size_type new_size, const T& value);
    /** Get capacity. Same as max_size(). */
    size_type capacity() const
        { return num_total; }
    /** Check whether vector is empty. */
    bool empty() const
        { return num_valid == 0; }

    /** Unchecked element access. */
    reference operator[](size_type n)
        { return data[n]; }
    /** Unchecked constant element access. */
    const_reference operator[](size_type n) const
        { return data[n]; }
    /** Checked element access. \throw std::range_error if n is out of range. */
    reference at(size_type n);
    /** Checked constant element access. \throw std::range_error if n is out of range. */
    const_reference at(size_type n) const;

    /** Access first element. */
    reference front()
        { return data[0]; }
    /** Access first element. */
    const_reference front() const
        { return data[0]; }

    /** Access last element. */
    reference back()
        { return data[num_valid-1]; }
    /** Access last element. */
    const_reference back() const
        { return data[num_valid-1]; }

    /** Append one element.
        \param x [in] Element to append
        \throw fixed_container_overrun if max_size() would be exceeded */
    void push_back(const T& x)
        {
            if (num_valid < num_total) {
                new (&data[num_valid]) T(x);
                ++num_valid;
            } else {
                throw fixed_container_overrun();
            }
        }
    /** Remove last element. */
    void pop_back()
        {
            ASSERT(num_valid > 0);
            data[--num_valid].~T();
        }

    /** Insert single element. Inserts \c x before iterator \c pos.
        \return Iterator pointing to inserted element */
    iterator insert(iterator pos, const T& x);
    /** Insert element repeatedly. Inserts \c n copies of \c x before
        \c pos.
        \return Iterator pointing to first inserted element. */
    void insert(iterator pos, size_type n, const T& x);
    /** Insert range. Inserts [first,last) before \c pos.
        \param first,last Range, forward iterators */
    template<typename It>
    void insert(iterator pos, It first, It last);

    /** Erase single element. Erases the element pointed to by \c pos.
        \return Iterator pointing to element after the erased one. */
    iterator erase(iterator pos);
    /** Erase range. Erases the range defined by [beg,end).
        \return Iterator pointing to element after the last erased one. */
    iterator erase(iterator beg, iterator end);

    /** Swap vectors. Exchanges two fixed_vector by swapping their
        individual elements.. */
    void swap(fixed_vector& other);

    /** Clear this vector. Destroys all elements.
        \post empty() */
    void clear();
 private:
    T* data;
    size_type num_valid;
    size_type num_total;

    void make_room(iterator it, size_type amount);
    void erase_n(iterator it, size_type amount);
};

template<typename T>
const std::size_t fixed_vector<T>::node_size;

template<typename T>
fixed_vector<T>&
fixed_vector<T>::operator=(const fixed_vector<T>& other)
{
    if (other.size() > max_size())
        throw fixed_container_overrun();

    clear();
    for (const_iterator i = other.begin(); i != other.end(); ++i)
        push_back(*i);
}

template<typename T>
template<typename It>
void
fixed_vector<T>::assign(It first, It last)
{
    this->clear();
    while (first != last)
        this->push_back(*first), ++first;
}

template<typename T>
void
fixed_vector<T>::assign(size_type n, const T& value)
{
    if (n > max_size())
        throw fixed_container_overrun();

    this->clear();
    while (n--)
        this->push_back(value);
}

template<typename T>
void
fixed_vector<T>::resize(size_type new_size, const T& value)
{
    if (new_size > this->max_size())
        throw fixed_container_overrun();
    while (this->size() < new_size)
        this->push_back(value);
    while (this->size() > new_size)
        this->pop_back();
}

template<typename T>
typename fixed_vector<T>::reference
fixed_vector<T>::at(size_type n)
{
    if (n > size())
        throw std::range_error("fixed_vector out of bounds access");
    return data[n];
}

template<typename T>
typename fixed_vector<T>::const_reference
fixed_vector<T>::at(size_type n) const
{
    if (n > size())
        throw std::range_error("fixed_vector out of bounds access");
    return data[n];
}

template<typename T>
typename fixed_vector<T>::iterator
fixed_vector<T>::insert(iterator pos, const T& x)
{
    if (num_valid >= num_total)
        throw fixed_container_overrun();

    if (pos != end()) {
        /* duplicate last element */
        push_back(end()[-1]);
        /* move other elements forward */
        for (iterator e = end()-1; e != pos; --e)
            e[0] = e[-1];
        *pos = x;
        return pos;
    } else {
        push_back(x);
        return pos;
    }
}

template<typename T>
void
fixed_vector<T>::insert(iterator pos, size_type n, const T& x)
{
    if (pos != end()) {
        make_room(pos, n);
        while (n--) {
            *pos = x;
            ++pos;
        }
    } else {
        resize(size() + n, x);
    }
}

template<typename T>
template<typename It>
void
fixed_vector<T>::insert(iterator pos, It first, It last)
{
    if (pos != end()) {
        size_type diff = std::distance(first, last);
        make_room(pos, diff);
        while (diff--) {
            *pos = *first;
            ++pos;
            ++first;
        }
    } else {
        while (first != last) {
            push_back(*first);
            ++first;
        }
    }
}

template<typename T>
inline typename fixed_vector<T>::iterator
fixed_vector<T>::erase(iterator pos)
{
    erase_n(pos, 1);
    return pos;
}

template<typename T>
inline typename fixed_vector<T>::iterator
fixed_vector<T>::erase(iterator beg, iterator end)
{
    erase_n(beg, end - beg);
    return beg;
}

template<typename T>
void
fixed_vector<T>::swap(fixed_vector& other)
{
    /* validate */
    if (this->num_valid > other.num_total)
        throw fixed_container_overrun();
    if (other.num_valid > this->num_total)
        throw fixed_container_overrun();

    /* swap common portion */
    size_type common = this->num_valid < other.num_valid ? this->num_valid : other.num_valid;
    for (size_type i = 0; i < common; ++i)
        std::swap(this->data[i], other.data[i]);

    if (this->num_valid > other.num_valid) {
        /* we are longer */
        while (other.num_valid < this->num_valid)
            other.push_back(this->data[other.num_valid]);
        while (this->size() > common)
            this->pop_back();
    } else if (other.num_valid > this->num_valid) {
        /* other container is longer */
        while (this->num_valid < other.num_valid)
            this->push_back(other.data[this->num_valid]);
        while (other.size() > common)
            other.pop_back();
    }
}

template<typename T>
void
fixed_vector<T>::clear()
{
    while (num_valid)
        data[--num_valid].~T();
}

template<typename T>
void
fixed_vector<T>::erase_n(iterator it, size_type n)
{
    size_type first_index = it - this->begin();
    size_type other_index = first_index + n;
    if (other_index > size())
        throw std::range_error("fixed_vector erase out of range");

    while (other_index < size())
        data[first_index++] = data[other_index++];
    while (first_index < size())
        pop_back();
}

template<typename T>
void
fixed_vector<T>::make_room(iterator it, size_type amount)
{
    if (size() + amount > max_size())
        throw fixed_container_overrun();
    for (size_type i = 0; i < amount; ++i)
        push_back(T());

    iterator in = end() - amount; // old end()
    iterator out = end();
    while (in != it)
        *--out = *--in;
}

template<typename T>
bool
operator==(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    if (lhs.size() != rhs.size())
        return false;
    for (typename fixed_vector<T>::size_type i = 0; i < lhs.size(); ++i)
        if (!(lhs[i] == rhs[i]))
            return false;
    return true;
}

template<typename T>
inline bool
operator!=(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    return !(lhs == rhs);
}

template<typename T>
inline bool
operator<=(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    return !(rhs < lhs);
}

template<typename T>
inline bool
operator>=(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    return !(lhs < rhs);
}

template<typename T>
bool
operator<(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    typename fixed_vector<T>::size_type cmp, i;
    cmp = lhs.size() < rhs.size() ? lhs.size() : rhs.size();
    for (i = 0; i < cmp; ++i) {
        if (lhs[i] < rhs[i])
            return true;
        if (rhs[i] < lhs[i])
            return false;
    }
    return lhs.size() < rhs.size();
}

template<typename T>
inline bool
operator>(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
{
    return rhs < lhs;
}

namespace std {
    template<class T>
    void swap(const fixed_vector<T>& lhs, const fixed_vector<T>& rhs)
    {
        lhs.swap(rhs);
    }
}

/******************* fixed_vector ready-made instances *******************/

/** Fixed-size vector with own storage. This is a fixed_vector which
    includes the necessary object storage in its object. */
template<typename T, std::size_t Amount>
class static_vector : public fixed_vector<T> {
    union {
        char data[Amount * sizeof(T)];
        typename min_sized_type<sizeof(T)>::type align;
    } data[1];
 public:
    typedef typename fixed_vector<T>::size_type size_type;

    static_vector()
        : fixed_vector<T>(data, Amount)
        { }

    static_vector(size_type n, const T& value)
        : fixed_vector<T>(data, Amount, n, value)
        { }

    template<typename It>
    static_vector(It first, It last)
        : fixed_vector<T>(data, Amount, first, last)
        { }

    static_vector(const fixed_vector<T>& other)
        : fixed_vector<T>(data, Amount, other)
        { }

    static_vector(const static_vector& other)
        : fixed_vector<T>(data, Amount, other)
        { }

    ~static_vector()
        {
            // Make sure that fixed_vector does not reference data
            // anymore
            this->clear();
        }
};

/** Fixed-size vector with heap storage. This is a fixed_vector which
    allocates its storage from the heap. Unlike with static_vector,
    you do not need to know the number of objects in advance. The
    heap_vector does not resize dynamically like std::vector, thus the
    only allocation happens in the constructor, and the only
    deallocation happens in the destructor. This always uses the
    global operator new / operator delete.

    Because you have to specify the object size, this class has no
    copy constructor. */
template<typename T>
class heap_vector : public fixed_vector<T> {
    void* allocate(std::size_t amount)
        {
            return ::operator new(amount * fixed_vector<T>::node_size);
        }
 public:
    typedef typename fixed_vector<T>::size_type size_type;

    /** Constructor. Makes an empty vector.
        \param amount Maximum number of elements. */
    explicit heap_vector(std::size_t amount)
        : fixed_vector<T>(allocate(amount), amount)
        { }

    /** Constructor. Makes a vector containing \c n copies of \c value.
        \param amount Maximum number of elements. */
    heap_vector(std::size_t amount, size_type n, const T& value)
        : fixed_vector<T>(allocate(amount), amount, n, value)
        { }

    /** Constructor. Makes a vector containing a copy of the range [first,last).
        \param amount Maximum number of elements. */
    template<typename It>
    heap_vector(std::size_t amount, It first, It last)
        : fixed_vector<T>(allocate(amount), amount, first, last)
        { }

    /** Constructor. Makes a copy of \c other.
        \param amount Maximum number of elements. */
    heap_vector(std::size_t amount, const fixed_vector<T>& other)
        : fixed_vector<T>(allocate(amount), amount, other)
        { }

    /** Destructor. */
    ~heap_vector()
        {
            this->clear();
            ::operator delete(&*this->begin());
        }
};

#endif
