/**
  *  \file fixed_list.h
  *  \brief Fixed-size List
  *  \author Stefan Reuther
  */
#ifndef FIXED_LIST_H_INCLUDED
#define FIXED_LIST_H_INCLUDED

#include <cstddef>
#include <algorithm>
#include <iterator>
#include "cpluslib/fixed_container.h"

/******************************* fixed_list ******************************/

template<typename T>
class heap_list;

/** Fixed-size List. This container provides a doubly-linked list that
    allocates nodes from a fixed-size pool of storage. Otherwise, the
    interface is as close as possible to std::list.

    Because you must pass pointers to the memory to manage into the
    constructor, this class has no copy constructor. All other
    functions remain the same as for list.

    Distinctive properties:
    - Objects can be inserted and deleted in constant time at any position
    - Provides bidirectional iterators
    - Linear memory management overhead

    Exception safety: if a copy constructor throws, you may be left
    with a partially-performed operation. The container itself will
    never get inconsistent.

    Overflow handling: like for exceptions, operations that overflow
    may be performed partially.

    Representation: this is a circular list. All user-created nodes
    are of type \c node, which has links and a value; the root node is
    of type \c link which has just the links. Iteration uses link
    pointers. */
template<typename T>
class fixed_list {
    /** Doubly-linked list node. */
    struct link {
        link* p[2];
        link*& next()
            { return p[0]; }
        link* next() const
            { return p[0]; }
        link*& prev()
            { return p[1]; }
        link* prev() const
            { return p[1]; }
        link()
            { p[0] = p[1] = this; }
    };
    /** Doubly-linked list node with value. Objects of this type are
        used for list elements. */
    struct node : link {
        T value;

        node(const T& t)
            : value(t)
            { }
    };

    /** Iterator.
        \param N  direction. 0=forward, 1=backward
        \param RT reference type. */
    template<int N, typename RT>
    struct node_it : public std::iterator<std::bidirectional_iterator_tag, RT> {
        // FIXME: iterator -> const_iterator conversions
        link* p;

        node_it(link* p)
            : p(p) { }
        node_it()
            : p(0) { }

        bool operator==(node_it other) const
            { return p == other.p; }
        bool operator!=(node_it other) const
            { return p != other.p; }
        RT& operator*() const
            { return static_cast<node*>(p)->value; }
        RT* operator->() const
            { return &static_cast<node*>(p)->value; }
        node_it& operator++()
            {
                p = p->p[N];
                return *this;
            }
        node_it operator++(int)
            {
                node* tmp = p;
                p = p->p[N];
                return node_it(tmp);
            }
        node_it& operator--()
            {
                p = p->p[!N];
                return *this;
            }
        node_it operator--(int)
            {
                node* tmp = p;
                p = p->p[!N];
                return node_it(tmp);
            }
    };

    fixed_node_set<node> data;
    link root;

    friend class heap_list<T>; // must access data to free it

 public:
    typedef T&              reference;
    typedef const T&        const_reference;
    typedef node_it<0,T>    iterator;
    typedef node_it<0,const T> const_iterator;
    typedef std::size_t     size_type;
    typedef std::ptrdiff_t  difference_type;
    typedef T               value_type;
    typedef T*              pointer;
    typedef const T*        const_pointer;
    typedef node_it<1,T>    reverse_iterator;
    typedef node_it<1,const T> const_reverse_iterator;

 public:
    /** Size of a list node. This is the amount of storage the user
        must provide for each object. For a fixed_list<T>, this is
        more than sizeof(T). */
    static const std::size_t node_size = fixed_node_set<node>::node_size;

    /** Construct empty list.
        \param data      Pointer to node_size*max_size bytes of sufficiently aligned memory.
        \param max_size  Maximum number of nodes this list will hold. */
    fixed_list(void* data, size_type max_size)
        : data(data, max_size)
        { }

    /** Construct list with initial elements. Initializes the list with
        multiple copies of an element.
        \param data      Pointer to node_size*max_size bytes of sufficiently aligned memory.
        \param max_size  Maximum number of nodes this list will hold.
        \param n         Number of elements to put into list
        \param value     Element to copy */
    fixed_list(void* data, size_type max_size, size_type n, const T& value = T())
        : data(data, max_size)
        {
            this->assign(n, value);
        }

    /** Construct list from range. Initializes the list with a copy of the
        specified range.
        \param data      Pointer to node_size*max_size bytes of sufficiently aligned memory.
        \param max_size  Maximum number of nodes this list will hold.
        \param first,last Range */
    template<typename It>
    fixed_list(void* data, size_type max_size, It first, It last)
        : data(data, max_size)
        {
            this->assign(first, last);
        }

    /** Construct list from other list. Initializes the list with a copy
        of all elements of another list.
        \param data      Pointer to node_size*max_size bytes of sufficiently aligned memory.
        \param max_size  Maximum number of nodes this list will hold.
        \param other     List to initialize from */
    fixed_list(void* data, size_type max_size, const fixed_list& other)
        : data(data, max_size)
        {
            this->operator=(other);
        }

    /** Destructor. */
    ~fixed_list()
        { clear(); }

    /** Assignment operator. */
    fixed_list& operator=(const fixed_list& other);

    /** Assign range. Replaces the list's contents with the contents
        of [first,last).
        \param first,last Range, input iterators */
    template<typename It>
    void assign(It first, It last);

    /** Assign value repeatedly. Replaces the list's contents with
        \c n copies of \c value. */
    void assign(size_type n, const T& value);

    /** Begin iterator. \return Modifyable bidirectional iterator to beginning. */
    iterator begin()
        { return iterator(root.next()); }
    /** Begin iterator. \return Constant bidirectional iterator to beginning. */
    const_iterator begin() const
        { return const_iterator(root.next()); }

    /** End iterator. \return Modifyable bidirectional iterator one-past-the-end. */
    iterator end()
        { return iterator(&root); }
    /** End iterator. \return Constant bidirectional iterator one-past-the-end. */
    const_iterator end() const
        // FIXME: get rid of this cast
        { return const_iterator(const_cast<link*>(&root)); }

    /** Begin iterator. \return Modifyable bidirectional reverse iterator to end. */
    reverse_iterator rbegin()
        { return reverse_iterator(root.prev()); }
    /** Begin iterator. \return Constant bidirectional reverse iterator to end. */
    const_reverse_iterator rbegin() const
        { return const_reverse_iterator(root.prev()); }

    /** End iterator. \return Modifyable bidirectional reverse iterator one-before-beginning. */
    reverse_iterator rend()
        { return reverse_iterator(&root); }
    /** End iterator. \return Constant bidirectional reverse iterator one-before-beginning. */
    const_reverse_iterator rend() const
        { return const_reverse_iterator(const_cast<link*>(&root)); }

    /** Check whether list is empty. */
    bool empty() const
        { return root.next() == &root; }
    /** Get size of list. \return current number of elements. */
    size_type size() const;
    /** Get maximum size. \return maximum number of elements, as set in the constructor. */
    size_type max_size() const
        { return data.max_size(); }
    /** Change size of vector. If \c new_size is less than size(),
        removes the excess elements. If \c new_size is larger than
        size(), enlarge the list by appending copies of \c value. */
    void resize(size_type sz, T value = T());

    /** Access first element. */
    reference front()
        { return static_cast<node*>(root.next())->value; }
    /** Access first element. */
    const_reference front() const
        { return static_cast<node*>(root.next())->value; }
    /** Access last element. */
    reference back()
        { return static_cast<node*>(root.prev())->value; }
    /** Access last element. */
    const_reference back() const
        { return static_cast<node*>(root.prev())->value; }

    /** Append one element.
        \param x [in] Element to append
        \throw fixed_container_overrun if max_size() would be exceeded */
    void push_back(const T& value);
    /** Prepend one element.
        \param x [in] Element to prepend
        \throw fixed_container_overrun if max_size() would be exceeded */
    void push_front(const T& value);
    /** Remove last element. */
    void pop_back();
    /** Remove first element. */
    void pop_front();

    /** Insert single element. Inserts \c x before iterator \c pos.
        \return Iterator pointing to inserted element */
    iterator insert(iterator pos, const T& x);
    /** Insert element repeatedly. Inserts \c n copies of \c x before
        \c pos. */
    void insert(iterator pos, size_type n, const T& x);
    /** Insert range. Inserts [first,last) before \c pos.
        \param first,last Range, forward iterators */
    template<typename It>
    void insert(iterator pos, It first, It last);

    /** Erase single element. Erases the element pointed to by \c pos.
        \return Iterator pointing to element after the erased one. */
    iterator erase(iterator pos);
    /** Erase range. Erases the range defined by [beg,end).
        \return Iterator pointing to element after the last erased one. */
    iterator erase(iterator pos, iterator last);

    /** Swap vectors. Exchanges two fixed_list by swapping their
        individual elements.. */
    void swap(fixed_list& other);

    /** Clear this list. Destroys all elements.
        \post empty() */
    void clear();

//     void splice(iterator pos, fixed_list& other);
//     void splice(iterator pos, fixed_list& other, iterator i);
//     void splice(iterator pos, fixed_list& other, iterator first, iterator last);

    /** Remove all copies of a value. Erases all elements that
        compare equal to \c value. */
    void remove(const T& value);
    /** Remove all values matching a predicate. Erases all elements
        for which pred(e) is true. */
    template<typename Pred>
    void remove_if(Pred pred);

    /** Remove successive duplicate elements. From each run of equal
        elements, erases all but the first one. */
    void unique();
    /** Remove successive duplicate elements. From each run of
        elements where bpred(first,next) returns true, erases all but
        the first one. */
    template<typename BPred>
    void unique(BPred bpred);

    /** Merge list into this one. Assuming this list and \c other are
        sorted ascendingly, merge \c other into this such that the
        result list is again sorted. For equal elements, those copied
        from this list come first.
        \post other.empty() */
    void merge(fixed_list& other)
        { this->merge(other, std::less<T>()); }
    /** Merge list into this one. Assuming this list and \c other are
        sorted ascendingly according to the comparison function \c
        compare, merge \c other into this such that the result list is
        again sorted. For equal elements, those copied from this list
        come first.
        \post other.empty() */
    template<typename Comp>
    void merge(fixed_list& other, Comp compare);

    /** Sort list ascendingly. */
    void sort()
        {
            if (!this->empty())
                this->sort_range(this->begin(), --this->end(), std::less<T>(), this->size()-1);
        }
    /** Sort list according to predicate. */
    template<typename Comp>
    void sort(Comp compare)
        {
            if (!this->empty())
                this->sort_range(this->begin(), --this->end(), compare, this->size()-1);
        }

    /** Reverse this list. This works by swapping the prev/next
        pointers, so no elements are copied. */
    void reverse();

 private:
    /** Sort subrange of list. This is a recursive off-the-book
        quicksort implementation. Quicksort requires comparison of
        array indices which list iterators do not provide. Therefore,
        the distance between the two iterators must be passed in to
        avoid that we have to recompute it.
        \param beg,end   Range. Unlike elsewhere, boundaries are inclusive.
        \param comp      Comparison function.
        \param range_len Difference between beg and end. */
    template<typename Comp>
    void sort_range(iterator beg, iterator end, const Comp& compare, size_type range_len);
};

template<typename T>
const std::size_t fixed_list<T>::node_size;

template<typename T>
typename fixed_list<T>::size_type
fixed_list<T>::size() const
{
    size_type result = 0;
    for (link* p = root.next(); p != &root; p = p->next())
        ++result;
    return result;
}

template<typename T>
fixed_list<T>&
fixed_list<T>::operator=(const fixed_list<T>& other)
{
    this->clear();
    for (const_iterator i = other.begin(); i != other.end(); ++i)
        this->push_back(*i);
    return *this;
}

template<typename T>
template<typename It>
void
fixed_list<T>::assign(It first, It last)
{
    this->clear();
    while (first != last)
        this->push_back(*first), ++first;
}

template<typename T>
void
fixed_list<T>::assign(size_type n, const T& value)
{
    this->clear();
    while (n--)
        this->push_back(value);
}

template<typename T>
void
fixed_list<T>::resize(size_type sz, T value)
{
    if (sz > this->max_size())
        throw fixed_container_overrun();

    size_type cur_size = this->size();
    while (cur_size > sz)
        this->pop_back(), --cur_size;
    while (cur_size < sz)
        this->push_back(value), ++cur_size;
}

template<typename T>
void
fixed_list<T>::push_front(const T& value)
{
    node* p = new(data.allocate_node()) node(value);
    p->prev() = &root;
    p->next() = root.next();
    p->next()->prev() = p;
    root.next() = p;
}

template<typename T>
void
fixed_list<T>::push_back(const T& value)
{
    node* p = new(data.allocate_node()) node(value);
    p->next() = &root;
    p->prev() = root.prev();
    p->prev()->next() = p;
    root.prev() = p;
}

template<typename T>
void
fixed_list<T>::pop_front()
{
    node* p = static_cast<node*>(root.next());
    root.next() = p->next();
    root.next()->prev() = &root;
    data.destroy_node(*p);
}

template<typename T>
void
fixed_list<T>::pop_back()
{
    node* p = static_cast<node*>(root.prev());
    root.prev() = p->prev();
    root.prev()->next() = &root;
    data.destroy_node(*p);
}

template<typename T>
typename fixed_list<T>::iterator
fixed_list<T>::insert(iterator pos, const T& x)
{
    node* n = new(this->data.allocate_node()) node(x);
    link* p = pos.p;
    n->next() = p;
    n->prev() = p->prev();
    n->prev()->next() = n;
    p->prev() = n;
    return iterator(n);
}

template<typename T>
void
fixed_list<T>::insert(iterator pos, size_type n, const T& x)
{
    while (n--)
        this->insert(pos, x);
}

template<typename T>
template<typename It>
void
fixed_list<T>::insert(iterator pos, It first, It last)
{
    while (first != last)
        pos = this->insert(*first), ++first;
}

template<typename T>
void
fixed_list<T>::clear()
{
    while (!this->empty())
        this->pop_back();
}

template<typename T>
typename fixed_list<T>::iterator
fixed_list<T>::erase(iterator pos)
{
    link* rv = pos.p->next();
    pos.p->prev()->next() = pos.p->next();
    pos.p->next()->prev() = pos.p->prev();
    data.destroy_node(*static_cast<node*>(pos.p));
    return iterator(rv);
}

template<typename T>
typename fixed_list<T>::iterator
fixed_list<T>::erase(iterator beg, iterator end)
{
    while (beg != end)
        beg = this->erase(beg);
    return beg;
}

template<typename T>
void
fixed_list<T>::swap(fixed_list<T>& other)
{
    iterator here = this->begin(), there = other.begin();
    while (here != this->end() && there != other.end()) {
        std::iter_swap(here, there);
        ++here;
        ++there;
    }
    if (here != this->end()) {
        while (here != this->end()) {
            other.push_back(*here);
            here = this->erase(here);
        }
    } else if (there != other.end()) {
        while (there != other.end()) {
            this->push_back(*there);
            there = other.erase(there);
        }
    }
}

template<typename T>
void
fixed_list<T>::remove(const T& value)
{
    iterator i = this->begin();
    while (i != this->end()) {
        if (*i == value)
            i = this->erase(i);
        else
            ++i;
    }
}

template<typename T>
template<typename Pred>
void
fixed_list<T>::remove_if(Pred pred)
{
    iterator i = this->begin();
    while (i != this->end()) {
        if (pred(*i))
            i = this->erase(i);
        else
            ++i;
    }
}

template<typename T>
void
fixed_list<T>::unique()
{
    iterator a = this->begin();
    if (a == this->end())
        return;
    iterator b = a;
    ++b;
    while (b != this->end()) {
        if (*b == *a) {
            b = this->erase(b);
        } else {
            ++b;
            ++a;
        }
    }
}

template<typename T>
template<typename BPred>
void
fixed_list<T>::unique(BPred bpred)
{
    iterator a = this->begin();
    if (a == this->end())
        return;
    iterator b = a;
    ++b;
    while (b != this->end()) {
        if (bpred(*b, *a)) {
            b = this->erase(b);
        } else {
            ++b;
            ++a;
        }
    }
}

template<typename T>
template<typename Comp>
void
fixed_list<T>::merge(fixed_list<T>& other, Comp compare)
{
    iterator here = this->begin(), there = other.begin();
    while (there != other.end() && here != this->end()) {
        if (compare(*there, *here)) {
            this->insert(here, *there);
            there = other.erase(there);
        } else {
            ++here;
        }
    }
    while (there != other.end()) {
        this->push_back(*there);
        ++there;
    }
}

template<typename T>
void
fixed_list<T>::reverse()
{
    for (link* p = root.next(); p != &root; p = p->prev())
        std::swap(p->next(), p->prev());
    std::swap(root.next(), root.prev());
}

template<typename T>
template<typename Comp>
void
fixed_list<T>::sort_range(iterator beg, iterator end, const Comp& compare, size_type range_len)
{
    iterator i = beg, j = end;
    difference_type ix = 0, jx = range_len;
    value_type x = *beg;
    do {
        while(compare(*i, x))
            ++i, ++ix;
        while(compare(x, *j))
            --j, --jx;
        if(ix < jx) {
            std::iter_swap(i, j);
            ++i, ++ix;
            --j, --jx;
        } else if(ix == jx) {
            if(i == end)
                --j, --jx;
            else
                ++i, ++ix;
        }
    } while(ix <= jx);
    if(0 < jx)
        this->sort_range(beg, j, compare, jx);
    if(ix < range_len)
        this->sort_range(i, end, compare, range_len - ix);
}

/****************************** static_list ******************************/

/** Fixed-size list with own storage. This is a fixed_list which
    includes the necessary object storage in its object. */
template<typename T, std::size_t Amount>
class static_list : public fixed_list<T> {
    union {
        char data[Amount * fixed_list<T>::node_size];
        typename min_sized_type<sizeof(T)>::type align;
    } data[1];
 public:
    typedef typename fixed_list<T>::size_type size_type;

    static_list()
        : fixed_list<T>(data, Amount)
        { }

    static_list(size_type n, const T& value)
        : fixed_list<T>(data, Amount, n, value)
        { }

    template<typename It>
    static_list(It first, It last)
        : fixed_list<T>(data, Amount, first, last)
        { }

    static_list(const fixed_list<T>& other)
        : fixed_list<T>(data, Amount, other)
        { }

    static_list(const static_list& other)
        : fixed_list<T>(data, Amount, other)
        { }

    ~static_list()
        {
            // Make sure that fixed_list does not reference data
            // anymore
            this->clear();
        }
};

/** Fixed-size vector with heap storage. This is a fixed_vector which
    allocates its storage from the heap. Unlike with static_vector,
    you do not need to know the number of objects in advance. The
    heap_vector does not resize dynamically like std::vector, thus the
    only allocation happens in the constructor, and the only
    deallocation happens in the destructor. This always uses the
    global operator new / operator delete.

    Because you have to specify the object size, this class has no
    copy constructor. */
template<typename T>
class heap_list : public fixed_list<T> {
    void* allocate(std::size_t amount)
        {
            return ::operator new(amount * fixed_list<T>::node_size);
        }
 public:
    typedef typename fixed_list<T>::size_type size_type;

    /** Constructor. Makes an empty list.
        \param amount Maximum number of elements. */
    explicit heap_list(std::size_t amount)
        : fixed_list<T>(allocate(amount), amount)
        { }

    /** Constructor. Makes a list containing \c n copies of \c value.
        \param amount Maximum number of elements. */
    heap_list(std::size_t amount, size_type n, const T& value)
        : fixed_list<T>(allocate(amount), amount, n, value)
        { }

    /** Constructor. Makes a list containing a copy of the range [first,last).
        \param amount Maximum number of elements. */
    template<typename It>
    heap_list(std::size_t amount, It first, It last)
        : fixed_list<T>(allocate(amount), amount, first, last)
        { }

    /** Constructor. Makes a copy of \c other.
        \param amount Maximum number of elements. */
    heap_list(std::size_t amount, const fixed_list<T>& other)
        : fixed_list<T>(allocate(amount), amount, other)
        { }

    /** Destructor. */
    ~heap_list()
        {
            this->clear();
            ::operator delete(this->data.data());
        }
};

#endif
