/**
  *  \file fixed_container.h
  *  \brief Fixed-size container utilities
  *  \author Stefan Reuther
  *
  *  This module provides a few things which are used by the actual
  *  container implementations.
  */
#ifndef FIXED_CONTAINER_H_INCLUDED
#define FIXED_CONTAINER_H_INCLUDED

#include <stdexcept>
#include <new>

/********************** Fixed Container Boilerplate **********************/

/** Compile-time "if-then-else" type switch.
    \param cond Condition
    \param T    Type to return if condition is true
    \param F    Type to return if condition is false */
template<bool cond, typename T, typename F>
struct if_then_else {
    /** Result type. Same as \c T if \c cond is true, otherwise same as \c F. */
    typedef T type;
};

template<typename T, typename F>
struct if_then_else<false,T,F> {
    typedef F type;
};

struct min_sized_type_aux {
    virtual void mem_fun_1();
    virtual void mem_fun_2();
    virtual ~min_sized_type_aux();
};

/** Type of minimum size. This returns the smallest type which has at
    least size \c N. The idea is to make sure that alignment
    constraints are fulfilled, i.e. that min_sized_type<sizeof(T)>::type
    has the same or stricter alignment requirements as T. */
template<std::size_t N>
struct min_sized_type {
    /** Result type. */
    typedef if_then_else<(N <= 1), char,
        if_then_else<(N <= sizeof(short)), short,
            if_then_else<(N <= sizeof(int)), int,
                if_then_else<(N <= sizeof(float)), float,
                    if_then_else<(N <= sizeof(long)), long,
                        if_then_else<(N <= sizeof(double)), double,
                            if_then_else<(N <= sizeof(long double)), long double,
                                if_then_else<(N <= sizeof(void*)), void*,
                                    if_then_else<(N <= sizeof(void(min_sized_type_aux::*)())), void(min_sized_type_aux::*)(),
                                        long double>
                                    >
                                >
                            >
                        >
                    >
                >
            >
        > type;
};

/** Fixed container overrun. This exception is thrown whenever a
    fixed-size container operation attempts to exceed the allowed
    size. */
class fixed_container_overrun : public std::exception {
 public:
    fixed_container_overrun()
        { }
    const char* what() const throw()
        { return "fixed container overrun"; }
};

/******************************** Node Set *******************************/

/** Fixed-size Set of Nodes. Provides a set of nodes of a given type
    which can be allocated and deallocated in any order.

    Note that this class keeps track of deallocated objects only.
    Users have to keep track of allocated items, and destroy them
    before the fixed_node_set is destroyed. Otherwise, destructors of
    the contained objects will not be called.

    \param T Type of Node Value */
template<typename T>
class fixed_node_set {
 public:
    /** Type for a node index or count. */
    typedef std::size_t size_type;
    /** Type of a node value. */
    typedef T           value_type;
 private:
    union node_type {
        node_type* next;
        char value[sizeof(value_type)];
    };
    node_type* nodes;
    node_type* free_list;
    size_type  num_used_nodes;
    size_type  num_total;

    fixed_node_set(const fixed_node_set&);
    fixed_node_set& operator=(const fixed_node_set&);
 public:
    /** Size of a node. This may be different from sizeof(value_type)
        because we must store a link pointer for memory management. */
    static const size_type node_size = sizeof(node_type);

    /** Constructor.
        \param data Memory to use for nodes. Must point to
                    num_total*node_size bytes of sufficiently aligned
                    memory.
        \param num_total Number of nodes to manage. */
    fixed_node_set(void* data, size_type num_total)
        : nodes(static_cast<node_type*>(data)), free_list(0), num_used_nodes(0), num_total(num_total)
        { }

    /** Allocate a node.
        \return pointer to uninitialized storage for an object of type T
        \throw fixed_container_overrun if no more free nodes remain */
    void* allocate_node();

    /** Destroy a node. Destroys the object and returns the node to the
        free node list.
        \param value reference to an initialized object of type T */
    void destroy_node(value_type& value);

    /** Free a node. Returns a node to the free node list whose value
        has already been destroyed.
        \param value pointer to storage */
    void free_node(void* value)
        {
            node_type& node = *static_cast<node_type*>(value);
            node.next = free_list;
            free_list = &node;
        }

    /** Get maximum size.
        \return the maximum number of allocate_node() calls that can
        succeed for this fixed_node_set. */
    size_type max_size() const
        { return num_total; }

    /** Get pointer to raw data. */
    void* data()
        { return nodes; }
};

template<typename T>
const typename fixed_node_set<T>::size_type fixed_node_set<T>::node_size;

template<typename T>
void*
fixed_node_set<T>::allocate_node()
{
    if (free_list) {
        node_type* n = free_list;
        free_list = free_list->next;
        return n;
    } else {
        if (num_used_nodes < num_total) {
            return &nodes[num_used_nodes++];
        } else {
            throw fixed_container_overrun();
        }
    }
}

template<typename T>
void
fixed_node_set<T>::destroy_node(value_type& value)
{
    value.~T();
    free_node(&value);
}

#endif
