/**
  *  \file call.h
  *  \brief Abstraction of Callable Objects (Closure)
  */
#ifndef STREU_CPLUSLIB_CALL_H_INCLUDED
#define STREU_CPLUSLIB_CALL_H_INCLUDED

#include "cpluslib/smartptr.h"

#define DECLARE_CALLABLE(NAME, ARGS, CALL, ABSARGS)                          \
/** Generic callable. This is a generalized function pointer.                \
    It can point to a null operation (only if return type is void),          \
    a static function, a bound member function, or a member function         \
    of an object pointer-to by a Ptr<>. */                                   \
class NAME {                                                                 \
    /** Handler base-class. */                                               \
    class Handler {                                                          \
     public:                                                                 \
        /** Virtual destructor. */                                           \
        virtual ~Handler() { }                                               \
        /** Perform the call. */                                             \
        virtual RV call(ARGS) = 0;                                           \
        /** Clone this object. */                                            \
        virtual Handler* clone() = 0;                                        \
    };                                                                       \
                                                                             \
    /** Static function handler. */                                          \
    class Static : public Handler {                                          \
        RV (*func)(ARGS);                                                    \
     public:                                                                 \
        Static(RV (*func)(ARGS)) : func(func) { }                            \
        RV call(ARGS) { return func(CALL); }                                 \
        Static* clone() { return new Static(*this); }                        \
    };                                                                       \
                                                                             \
    /** Bound member function handler. */                                    \
    template<class T>                                                        \
    class Object : public Handler {                                          \
        T* obj;                                                              \
        RV (T::*func)(ARGS);                                                 \
     public:                                                                 \
        Object(T* obj, RV (T::*func)(ARGS)) : obj(obj), func(func) { }       \
        RV call(ARGS) { return (obj->*func)(CALL); }                         \
        Object* clone() { return new Object(*this); }                        \
    };                                                                       \
                                                                             \
    /** Bound member function with smart pointer. */                         \
    template<class T>                                                        \
    class SmartObj : public Handler {                                        \
        Ptr<T> obj;                                                          \
        RV (T::*func)(ARGS);                                                 \
     public:                                                                 \
        SmartObj(Ptr<T> obj, RV (T::*func)(ARGS)) : obj(obj), func(func) { } \
        RV call(ARGS) { return ((*obj).*func)(CALL); }                       \
        SmartObj* clone() { return new SmartObj(*this); }                    \
    };                                                                       \
                                                                             \
    static void nullop(ABSARGS) { }                                          \
                                                                             \
    Handler* handler;                                                        \
 public:                                                                     \
    /** Create pointer to null operation. Only allowed                       \
        if RV is void. */                                                    \
    NAME()                                                                   \
        : handler(new Static(nullop)) { }                                    \
                                                                             \
    /** Create pointer to static function. */                                \
    NAME(RV (*func)(ARGS))                                                   \
        : handler(new Static(func)) { }                                      \
                                                                             \
    /** Create pointer to bound member function. */                          \
    template<class T>                                                        \
    NAME(T& obj, RV (T::*func)(ARGS))                                        \
        : handler(new Object<T>(&obj, func)) { }                             \
                                                                             \
    /** Create pointer to bound member of smart-pointer. */                  \
    template<class T>                                                        \
    NAME(Ptr<T> obj, RV (T::*func)(ARGS))                                    \
        : handler(new SmartObj<T>(obj, func)) { }                            \
                                                                             \
    /** Copy this pointer. */                                                \
    NAME(const NAME& rhs)                                                    \
        : handler(rhs.handler->clone()) { }                                  \
                                                                             \
    /** Destructor. */                                                       \
    ~NAME() { delete handler; }                                              \
                                                                             \
    /** Copy this pointer. */                                                \
    NAME& operator=(const NAME& rhs) {                                       \
        if (rhs.handler != handler) {                                        \
            Handler* n = rhs.handler->clone();                               \
            delete handler;                                                  \
            handler = n;                                                     \
        }                                                                    \
        return *this;                                                        \
    }                                                                        \
                                                                             \
    /** Call the function pointed to by this object. */                      \
    RV operator()(ARGS) const { return handler->call(CALL); }                \
}

#define EMPTY
#define AND ,

template<class RV>
DECLARE_CALLABLE(Callable0, EMPTY, EMPTY, EMPTY);
template<class RV, class A>
DECLARE_CALLABLE(Callable1, A a, a, A);
template<class RV, class A, class B>
DECLARE_CALLABLE(Callable2, A a AND B b, a AND b, A AND B);

#undef DECLARE_CALLABLE

#define DECLARE_MAKE(RET, TPLARGS, ARGS)                                \
template<class RV TPLARGS>                                              \
RET makeCallable(RV (*fun)(ARGS)) { return RET(fun); }                  \
template<class RV TPLARGS, class T>                                     \
RET makeCallable(T& o, RV (T::*fun)(ARGS)) { return RET(o, fun); }      \
template<class RV TPLARGS, class T>                                     \
RET makeCallable(Ptr<T> o, RV (T::*fun)(ARGS)) { return RET(o, fun); }

DECLARE_MAKE(Callable0<RV>,             EMPTY,                   EMPTY)
DECLARE_MAKE(Callable1<RV AND A>,       AND class A,             A)
DECLARE_MAKE(Callable2<RV AND A AND B>, AND class A AND class B, A AND B)

#undef DECLARE_MAKE
#undef AND
#undef EMPTY

#endif
