/**
  *  \file cpluslib/buffer.h
  *  \brief Growable buffer
  *  \author Stefan Reuther
  */

#ifndef STREU_CPLUSLIB_BUFFER_H_INCLUDED
#define STREU_CPLUSLIB_BUFFER_H_INCLUDED

#include <algorithm>
#include <cstddef>
#include "cpluslib/smartptr.h"

/** Growable buffer. This buffer is an easy, simple, fast container
    for small objects like chars. It will most likely perform very bad
    for items which are not PODs. */
template<class T>
class Buffer {
    std::size_t cap;
    std::size_t length;
    T* buffer;
    Buffer(const Buffer&);
    Buffer& operator=(const Buffer&);
    enum {
        DEFAULT_CAPACITY = 128,
        MINIMUM_CAPACITY = 128
    };
 public:
    /** Constructor. */
    Buffer(std::size_t initial_capacity = DEFAULT_CAPACITY)
        {
            if(initial_capacity < MINIMUM_CAPACITY)
                initial_capacity = MINIMUM_CAPACITY;
            cap = initial_capacity;
            length = 0;
            buffer = new T[initial_capacity];
        }

    /** Destructor. */
    ~Buffer()
        { delete[] buffer; }

    /** Ensure capacity is at least /new_cap/. This might reallocate
        the buffer and invalidates all iterators / pointers.  */
    void ensureCapacity(std::size_t new_cap)
        {
            if(cap >= new_cap)
                return;
            if(new_cap < length*2)
                new_cap = length*2;

            T* n = new T[new_cap];  // may throw
            try {
                std::copy(buffer, buffer + length, n);
                delete[] buffer;
                buffer = n;
                cap = new_cap;
            }
            catch(...) {            // can only come from copy
                delete[] n;
                throw;
            }
        }

    /** Append array of items.
        \param what  [in] pointer to items
        \param count [in] number of items */
    void append(const T what[], std::size_t count)
        {
            ensureCapacity(length + count);
            std::copy(what, what + count, buffer + length);
            length += count;
        }

    /** Append one item. */
    void append(const T& what)
        { append(&what, 1); }

    /** Append item, replicated.
        \param what  item to append
        \param count number of times to append that item. */
    void appendN(const T& what, std::size_t count)
        {
            ensureCapacity(length + count);
            std::fill_n(buffer + length, count, what);
            length += count;
        }

    /** Reset buffer to zero lenght. This will not free the memory it
        uses. */
    void clear()
        { length = 0; }

    /** Get current capacity. */
    std::size_t getCapacity() const
        { return cap; }

    /** Get current size. */
    std::size_t getSize() const
        { return length; }

    /** Ensure size. This will reallocate the buffer if needed, but
        not explicitly initialize the new items. */
    void ensureSize(std::size_t s)
        {
            if(s <= length)
                return;
            ensureCapacity(s);
            length = s;
        }

    /** Swap two buffers. */
    void swap(Buffer& rhs)
        {
            std::size_t rcap    = rhs.cap;
            std::size_t rlength = rhs.length;
            T* rbuffer          = rhs.buffer;
            rhs.cap             = cap;
            rhs.length          = length;
            rhs.buffer          = buffer;
            cap                 = rcap;
            length              = rlength;
            buffer              = rbuffer;
        }

    /** Extract array. This is a destructive operation, the ArrayPtr
        will receive ownership over the old data, the buffer will
        afterwards be empty. */
    ArrayPtr<T> getArray()
        {
            ArrayPtr<T> val = buffer;
            cap = DEFAULT_CAPACITY;
            length = 0;
            buffer = new T[cap];
            return val;
        }

    /** Indexed access. */
    const T& operator[](std::size_t i) const
        { return buffer[i]; }

    T& operator[](std::size_t i)
        { return buffer[i]; }

    /** Get pointer to data. */
    const T* ptr() const
        { return buffer; }

    T* ptr()
        { return buffer; }

    /** Get iterator (pointer) to beginning. */
    const T* begin() const
        { return buffer; }

    T* begin()
        { return buffer; }

    /** Get iterator (pointer) to end. */
    const T* end() const
        { return buffer + length; }

    T* end()
        { return buffer + length; }
};

/** Scoped variable-size array. This class is a dynamically-allocated,
    variable-size, non-sharable array which is automatically deleted
    when it goes out of scope. The idea is to mimic C99's VLA feature
    to some extent. */
template<class T>
class Array {
    Array& operator=(const Array&);
    Array(const Array&);

    T*const data;
 public:
    /** Constructor. Creates an array of the specified size. */
    Array(std::size_t size)
        : data(new T[size]) { }

    /** Destructor. */
    ~Array()
        { delete[] data; }

    /** Element access. */
    const T& operator[](std::size_t i) const
        { return data[i]; }

    T& operator[](std::size_t i)
        { return data[i]; }

    /** Conversion to pointer. */
    operator const T*() const
        { return data; }

    operator T*()
        { return data; }

    const T* ptr() const
        { return data; }

    T* ptr()
        { return data; }
};

#endif
