/**
  *  \file bitops.h
  *  \brief Bit Operations
  *  \author Stefan Reuther
  */
#ifndef STREU_CPLUSLIB_BITOPS_H_INCLUDED
#define STREU_CPLUSLIB_BITOPS_H_INCLUDED

#include "cpluslib/types.h"

/** Left-rotate an 8-bit integer. */
inline uint8
rol8(uint8 a, int c)
{
    register unsigned int aa = a;
    aa <<= (c & 7);
    return (aa & 0xFFU) | (aa >> 8);
}

/** Right-rotate an 8-bit integer. */
inline uint8
ror8(uint8 a, int c)
{
    return rol8(a,8-c);
}

/** Left-rotate a 16-bit integer. */
inline uint16
rol16(uint16 a, int c)
{
    register unsigned int aa = a;
    aa <<= (c & 15);
    return (aa & 0xFFFFU) | (aa >> 16);
}

/** Right-rotate a 16-bit integer. */
inline uint16
ror16(uint16 a, int c)
{
    return rol16(a,16-c);
}

/** Left-rotate a 32-bit integer.
    gcc/i386 correctly collapses this weird C construct into a
    single CPU instruction for constant c's ... cool. */
inline uint32
rol32(uint32 a, int c)
{
    c &= 31;
    return (a << c) | ((a >> (32-c)) & 0xFFFFFFFFU);
}

/** Right-rotate a 32-bit integer. */
inline uint32
ror32(uint32 a, int c)
{
    c &= 31;
    return (a << (32-c)) | ((a >> c) & 0xFFFFFFFFU);
}

/** true if i is a power of two. Templatized to avoid unneeded
    widening of the operands. */
template<class T>
inline bool
isPowerOfTwo(T i)
{
    return (i & (i-1)) == 0;
}

#endif
