/**
  *  \file cpluslib/assert.h
  *  \brief Static Checks
  *  \author Stefan Reuther
  */

#ifndef STREU_CPLUSLIB_ASSERT_H_INCLUDED
#define STREU_CPLUSLIB_ASSERT_H_INCLUDED

#include "cpluslib/config.h"

/*! \def ASSERT
    \brief Assertion

    In debug build, tests the expression \c what, and terminates the
    program with assertionFailed() if the expression is false. In a
    release build, the expression \c what will not be evaluated, use
    VALIDATE() instead. */

/*! \def FAIL
    \brief Failure

    Shorthand for ASSERT(0), i.e. always terminates the program if
    this statement is reached. In release build, does nothing. Use
    this to document places of code which are not reached. */

/*! \def FIXME
    \brief Failure

    Shorthand for ASSERT(0), i.e. always terminates the program if
    this statement is reached. Unlike FAIL(), however, the program
    will not compile in release build. Use this to document places of
    incomplete code which you expect to complete for the release.
    \param what a string literal which will be displayed with the
    error message */

/*! \def VALIDATE
    \brief Assertion

    Like ASSERT(), but also evaluates the expression in release build.
    You should probably avoid this macro and do the evaluation
    explicitly. */

#ifdef STREU_CONFIG_DEBUG
#  define ASSERT(what) ((void)((what) ? 0 : assertionFailed(#what, __FILE__, __LINE__)))
#  define FAIL()       assertionFailed("FAIL", __FILE__, __LINE__)
#  define FIXME(what)  assertionFailed("FIXME: " what, __FILE__, __LINE__)
#  define VALIDATE(what) ASSERT(what)
/** Display assertion failure. cpluslib provides a default version of
    this function but you can also define your own. */
int assertionFailed(const char* what, const char* file, int line);
#else
#  define ASSERT(what) ((void) 0)
#  define FAIL()       ((void) 0)
#  define FIXME(what)  THIS_NEEDS_TO_BE_FIXED
#  define VALIDATE(what) ((void) (what))
#endif

/** Compile-time assertion. Your code will not compile if the
    compile-time constant \c expr evaluates to false. The error
    message will complain about a negative array size. This macro can
    be used in namespace and block scope. */
#define CTASSERT(expr) extern int ctassertArray[(expr) ? 2 : -2]

/** Checked pointer cast. checked_cast<T>(s) is the same as
    dynamic_cast<T>(s), but ensures that the return value is
    nonzero. */
template<class T, class S>
inline T
checked_cast(S s) {
    T t = dynamic_cast<T>(s);
    ASSERT(t);
    return t;
}

#endif
