/**
  *  \file array.h
  *  \brief Array Wrapper
  *  \author Stefan Reuther
  *
  *  These classes provide wrappers for 1-D, 2-D and 3-D arrays. Its
  *  main purpose is to fix the association between the array and its
  *  dimensions which usually gets lost when the array is passed to a
  *  function. In addition, it provides functions to logically modify
  *  the layout of the array, i.e. taking a subrange, modifying the
  *  stride or transposing it.
  *
  *  Possible improvements:
  *  - direct column access. Now, you do
  *    two_dee.transpose_xy()[0] to access just the first column
  *    of an array.
  *  - optional bounds checking
  *  - STL iterators
  *
  *  Things which cannot be done with this interface:
  *  - remove the X stride and hardcode it to 1. This simplifies the
  *    assembler code, but rules out transposition
  *  - offer an operation to turn a 2-D array into a 1-D array. This
  *    fails in case the original 2-D array is transposed or an
  *    X subrange is taken.
  *
  *  The stride is deliberately not published through the interface. 
  *  It remains to debate whether it should. Pro: it would allow users
  *  to write special-case handling code, if that would help them. 
  *  Con: it would make it too easy and tempting to use it when it is
  *  not needed. If it is desperately needed, you can still take the
  *  difference of the addresses of two elements.
  *
  *  \todo constness handling is not yet very well-thought-out.
  */

#ifndef STREU_CPLUSLIB_ARRAY_H_INCLUDED
#define STREU_CPLUSLIB_ARRAY_H_INCLUDED

#include <cstddef>

template <typename T> class array2;
template <typename T> class array3;

/** Divide p/a and round up. */
inline std::ptrdiff_t div_ceil(std::ptrdiff_t p, std::size_t a)
{ return (p + a-1) / a; }

/** One-dimensional array. This is a wrapper for a one-dimensional
    array which firmly associates a pointer and a size. Instead of
    passing pointer and size separately, pass array1<T> objects
    instead. In addition, this class supports modifying the size and
    stride of the array on the fly.

    array1<> does no memory management. The user is responsible for
    ensuring the lifetime of the stored pointer.

    array1<> does no range checking. */
template <typename T>
class array1 {
    T* ptr;
    std::size_t    size1;
    std::ptrdiff_t stride1;
 public:
    /** Constructor. Creates wrapper for a directly-allocated
        array. */
    template<std::size_t i>
    array1(T (&a)[i])
        : ptr(a), size1(i), stride1(1) { }

    /** Constructor. Create wrapper with explicitly-given
        parameters. */
    array1(T* ptr, std::size_t size1, std::ptrdiff_t stride1)
        : ptr(ptr), size1(size1), stride1(stride1) { }

    /** Get first dimension. */
    std::size_t size_x() const
        { return size1; }

    /** Element access.
        \pre index \in [0, size_x()) */
    T& operator[](std::size_t index) const
        { return ptr[stride1 * index]; }

    /** Create X subrange.
        \pre start <= end, start,end \in [0,size_x()) */
    const array1<T> subrange_x(std::size_t start, std::size_t end) const
        { return array1<T>(ptr + stride1*start, end - start, stride1); }

    /** Create sub-array with different stride. */
    const array1<T> with_stride_x(std::size_t new_stride) const
        { return array1<T>(ptr, div_ceil(size1, new_stride), stride1 * new_stride); }

    /** Convert to array-of-const. */
    operator array1<const T>() const
        { return array1<const T>(ptr, size1, stride1); }

    const array2<T> make_2d() const;
    const array2<T> make_2d(std::size_t size) const;

    const array3<T> make_3d() const;
    const array3<T> make_3d(std::size_t size1, std::size_t size2) const;
};

/** Two-dimensional array. This is a wrapper for a two-dimensional
    array which firmly associates a pointer and the sizes. Instead of
    passing pointer and size separately, pass array2<T> objects
    instead. In addition, this class supports modifying the size and
    stride of the array on the fly, as well as transposing it
    virtually.

    There are two dimensions, X (columns) and Y (rows), which
    correspond to the usual array semantics of C++: Y is the first
    index, X is the second one. The Y stride defines the offset
    between two array rows, it can be larger or smaller than X-stride
    times X-size.

    array2<> does no memory management. The user is responsible for
    ensuring the lifetime of the stored pointer.

    array2<> does no range checking. */
template <typename T>
class array2 {
    T* ptr;
    std::size_t    size1, size2;
    std::ptrdiff_t stride1, stride2;
 public:
    /** Constructor. Construct wrapper for directly-allocated
        array. */
    template<std::size_t i, std::size_t j>
    array2(T (&a)[i][j])
        : ptr(a[0]), size1(i), size2(j), stride1(j), stride2(1) { }

    /** Constructor. Construct wrapper from explicitly-given
        parameters.
        \param ptr            pointer to array data
        \param size1,stride1  first dimension (Y)
        \param size2,stride2  second dimension (X) */
    array2(T* ptr,
           std::size_t size1, std::ptrdiff_t stride1,
           std::size_t size2, std::ptrdiff_t stride2)
        : ptr(ptr), size1(size1), size2(size2),
          stride1(stride1), stride2(stride2) { }

    /** Get second dimension (X). */
    std::size_t size_x() const
        { return size2; }

    /** Get first dimension (Y). */
    std::size_t size_y() const
        { return size1; }

    /** Row access. Returns handle to the specified row. */
    const array1<T> operator[](std::size_t index) const
        { return array1<T>(ptr + stride1 * index, size2, stride2); }

    /** Create X subrange. Returns a new array containing just the
        given columns [start,end). */
    const array2<T> subrange_x(std::size_t start, std::size_t end) const
        { return array2<T>(ptr + stride2*start, size1, stride1, end - start, stride2); }

    /** Create Y subrange. Returns a new array containing just the
        given rows [start,end). */
    const array2<T> subrange_y(std::size_t start, std::size_t end) const
        { return array2<T>(ptr + stride1*start, end - start, stride1, size2, stride2); }

    /** Create sub-array with given X-stride. Returns a new array
        which contains only every /new_stride/th column, starting with
        the first (index zero). */
    const array2<T> with_stride_x(std::size_t new_stride) const
        { return array2<T>(ptr, size1, stride1, div_ceil(size2, new_stride), stride2 * new_stride); }

    /** Create sub-array with given Y-stride. Returns a new array
        which contains only every /new_stride/th row, starting with
        the first (index zero). */
    const array2<T> with_stride_y(std::size_t new_stride) const
        { return array2<T>(ptr, div_ceil(size1, new_stride), stride1 * new_stride, size2, stride2); }

    /** Transpose X and Y. Returns a new array with both indices
        swapped. */
    const array2<T> transposed_xy() const
        { return array2<T>(ptr, size2, stride2, size1, stride1); }

    operator array2<const T>() const
        { return array2<const T>(ptr, size1, stride1, size2, stride2); }

    const array3<T> make_3d() const;
};

/** Three-dimensional array. This is a wrapper for a three-dimensional
    array which firmly associates a pointer and the sizes. Instead of
    passing pointer and size separately, pass array3<T> objects
    instead. In addition, this class supports modifying the size and
    stride of the array on the fly, as well as transposing it
    virtually.

    There are three dimensions, X (columns), Y (rows), and Z (layers),
    which correspond to the usual array semantics of C++: Z is the
    first index, Y is the second, X is the third one. The Y stride
    defines the offset between two array rows, it can be larger or
    smaller than X-stride times X-size. Likewise, Z stride is the
    offset between two layers.

    array3<> does no memory management. The user is responsible for
    ensuring the lifetime of the stored pointer.

    array3<> does no range checking. */
template <typename T>
class array3 {
    T* ptr;
    std::size_t    size1, size2, size3;
    std::ptrdiff_t stride1, stride2, stride3;
 public:
    /** Constructor. Construct wrapper for directly-allocated
        array. */
    template<std::size_t i, std::size_t j, std::size_t k>
    array3(T (&a)[i][j][k])
        : ptr(a[0][0]), size1(i), size2(j), size3(k), stride1(j*k), stride2(k), stride3(1) { }

    /** Constructor. Construct wrapper from explicitly-given
        parameters.
        \param ptr            pointer to array data
        \param size1,stride1  first dimension (Z)
        \param size2,stride2  second dimension (Y)
        \param size3,stride3  third dimension (X) */
    array3(T* ptr,
           std::size_t size1, std::ptrdiff_t stride1,
           std::size_t size2, std::ptrdiff_t stride2,
           std::size_t size3, std::ptrdiff_t stride3)
        : ptr(ptr), size1(size1), size2(size2), size3(size3),
          stride1(stride1), stride2(stride2), stride3(stride3) { }

    /** Get third dimension (X). */
    std::size_t size_x() const
        { return size3; }
        
    /** Get second dimension (Y). */
    std::size_t size_y() const
        { return size2; }

    /** Get first dimension (Z). */
    std::size_t size_z() const
        { return size1; }

    /** Layer access. Returns the /index/th layer, which can be
        further subscripted. */
    const array2<T> operator[](std::size_t index) const
        { return array2<T>(ptr + stride1 * index, size2, stride2, size3, stride3); }

    /** Create X subrange. Returns a new array containing just the
        given columns [start,end). */
    const array3<T> subrange_x(std::size_t start, std::size_t end) const
        { return array3<T>(ptr + stride3*start, size1, stride1, size2, stride2, end - start, stride3); }

    /** Create Y subrange. Returns a new array containing just the
        given lines [start,end). */
    const array3<T> subrange_y(std::size_t start, std::size_t end) const
        { return array3<T>(ptr + stride2*start, size1, stride1, end - start, stride2, size3, stride3); }

    /** Create Z subrange. Returns a new array containing just the
        given layers [start,end). */
    const array3<T> subrange_z(std::size_t start, std::size_t end) const
        { return array3<T>(ptr + stride1*start, end - start, stride1, size2, stride2, size3, stride3); }

    /** Create sub-array with given X-stride. Returns a new array
        which contains only every /new_stride/th column, starting with
        the first (index zero). */
    const array3<T> with_stride_x(std::size_t new_stride) const
        { return array3<T>(ptr, size1, stride1, size2, stride2, div_ceil(size3, new_stride), stride3 * new_stride); }

    /** Create sub-array with given Y-stride. Returns a new array
        which contains only every /new_stride/th line, starting with
        the first (index zero). */
    const array3<T> with_stride_y(std::size_t new_stride) const
        { return array3<T>(ptr, size1, stride1, div_ceil(size2, new_stride), stride2 * new_stride, size3, stride3); }

    /** Create sub-array with given Z-stride. Returns a new array
        which contains only every /new_stride/th layer, starting with
        the first (index zero). */
    const array3<T> with_stride_z(std::size_t new_stride) const
        { return array3<T>(ptr, div_ceil(size1, new_stride), stride1 * new_stride, size2, stride2, size3, stride3); }

    /** Transpose X and Y. Returns a new array which has X and Y
        swapped, such that
        <pre>a[z][y][x] == a.transposed_xy()[z][x][y]</pre>. */
    const array3<T> transposed_xy() const
        { return array3<T>(ptr, size1, stride1, size3, stride3, size2, stride2); }

    /** Transpose X and Z. Returns a new array which has X and Z
        swapped, such that
        <pre>a[z][y][x] == a.transposed_xz()[x][y][z]</pre>. */
    const array3<T> transposed_xz() const
        { return array3<T>(ptr, size3, stride3, size2, stride2, size1, stride1); }

    /** Transpose Y and Z. Returns a new array which has Y and Z
        swapped, such that
        <pre>a[z][y][x] == a.transposed_xy()[y][z][x]</pre>. */
    const array3<T> transposed_yz() const
        { return array3<T>(ptr, size2, stride2, size1, stride1, size3, stride3); }

    operator array3<const T>() const
        { return array3<const T>(ptr, size1, stride1, size2, stride2, size3, stride3); }
};

/** Convert to 2-D array. Returns a 2-D array which has one row
    containing this array. */
template<typename T>
const array2<T>
array1<T>::make_2d() const
{
    return array2<T>(ptr, 1, 1, this->size1, this->stride1);
}

/** Convert to 2-D array. Returns a 2-D array which consists of rows
    of the given size. The number of rows is implicitly derived from
    the current size. */
template<typename T>
const array2<T>
array1<T>::make_2d(std::size_t size) const
{
    return array2<T>(ptr, this->size1 / size, this->stride1 * size, size, this->stride1);
}

/** Convert to 3-D array. Returns 3-D array which has one layer
    containing one row containing this array. */
template<typename T>
const array3<T>
array1<T>::make_3d() const
{
    return array3<T>(ptr, 1, 1, 1, 1, this->size1, this->stride1);
}

/** Convert to 3-D array. Returns 3-D array which contains rows/layers
    of the given sizes, the number of layers is derived from the
    current size.
    \param size1 number of rows per layer
    \param size2 number of elements per row */
template<typename T>
const array3<T>
array1<T>::make_3d(std::size_t size1, std::size_t size2) const
{
    return array3<T>(ptr,
                     this->size1 / (size1*size2), this->stride1 * size1 * size2,
                     size1, this->stride1 * size2,
                     size2, this->stride1);
}

/** Convert to 3-D array. Returns 3-D array with one layer, containing
    this array. */
template<typename T>
const array3<T>
array2<T>::make_3d() const
{
    return array3<T>(ptr, 1, 1, size2, stride2, size1, stride1);
}

#endif
