/**
  *  \file test/interpreter/fusiontest.cpp
  *  \brief Test for interpreter::Fusion
  */

#include "interpreter/fusion.hpp"

#include "afl/test/testrunner.hpp"
#include "interpreter/bytecodeobject.hpp"

using interpreter::Opcode;
using interpreter::BytecodeObject;

namespace {
    bool isInstruction(const Opcode& insn, const Opcode::Major major, uint8_t minor)
    {
        return insn.major == major
            && insn.minor == minor;
    }

    bool isInstruction(const Opcode& insn, const Opcode::Major major, uint8_t minor, uint16_t arg)
    {
        return isInstruction(insn, major, minor)
            && insn.arg == arg;
    }
}

/*
 *  Test fusion push+binary
 */

// pushloc + binary -> fused
AFL_TEST("interpreter.Fusion:fused:pushloc+binary", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    bco.addInstruction(Opcode::maBinary, interpreter::biAdd, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 2U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedBinary, Opcode::sLocal, 3));
    a.check("insn 1", isInstruction(bco(1), Opcode::maBinary,      interpreter::biAdd, 0));
}

// pushvar + binary -> not fused
AFL_TEST("interpreter.Fusion:kept:pushvar+binary", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush, Opcode::sNamedVariable, 3);
    bco.addInstruction(Opcode::maBinary, interpreter::biAdd, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 2U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maPush,        Opcode::sNamedVariable, 3));
    a.check("insn 1", isInstruction(bco(1), Opcode::maBinary,      interpreter::biAdd, 0));
}

/*
 *  Test fusion push+unary.
 */

// pushgvar + unary -> fused
AFL_TEST("interpreter.Fusion:fused:pushgvar+unary", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sNamedShared, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unStr, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 2U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sNamedShared, 7));
    a.check("insn 1", isInstruction(bco(1), Opcode::maUnary,      interpreter::unStr, 0));
}

// pushint + unary -> not fused
AFL_TEST("interpreter.Fusion:kept:pushint+unary", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sInteger, 9);
    bco.addInstruction(Opcode::maUnary, interpreter::unAtomStr, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 2U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maPush,  Opcode::sInteger, 9));
    a.check("insn 1", isInstruction(bco(1), Opcode::maUnary, interpreter::unAtomStr, 0));
}

/*
 *  Test fusion push+unary to in-place operation.
 */

// pushloc + uinc + poploc -> in-place
AFL_TEST("interpreter.Fusion:in-place:pushloc+unary+poploc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 3U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maInplaceUnary, Opcode::sLocal, 7));
    a.check("insn 1", isInstruction(bco(1), Opcode::maUnary, interpreter::unInc, 0));
    a.check("insn 2", isInstruction(bco(2), Opcode::maPop, Opcode::sLocal, 7));
}

// pushloc + uinc + pushloc -> fused, not in-place [value re-used]
AFL_TEST("interpreter.Fusion:fused:pushloc+unary+pushloc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 3U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
    a.check("insn 1", isInstruction(bco(1), Opcode::maUnary, interpreter::unInc, 0));
    a.check("insn 2", isInstruction(bco(2), Opcode::maPush, Opcode::sLocal, 7));
}

// pushloc + uinc -> fused, not in-place [value not provably overwritten]
AFL_TEST("interpreter.Fusion:fused:pushloc+unary", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 2U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
    a.check("insn 1", isInstruction(bco(1), Opcode::maUnary, interpreter::unInc, 0));
}

// pushloc + uinc + other + poploc -> in-place [overwritten after other operations]
AFL_TEST("interpreter.Fusion:in-place:pushloc+unary+other+poploc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maUnary, interpreter::unStr, 0);
    bco.addInstruction(Opcode::maBinary, interpreter::biMult, 0);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 5U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maInplaceUnary, Opcode::sLocal, 7));
}

// catch + pushloc + uinc + other + poploc -> fused, not in-place [overwritten after other operations, but not exception-safe]
AFL_TEST("interpreter.Fusion:fused:catch+pushloc+unary+other+poploc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maJump,  Opcode::jCatch, 1);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maUnary, interpreter::unStr, 0);
    bco.addInstruction(Opcode::maBinary, interpreter::biMult, 0);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 6U);
    a.check("insn 1", isInstruction(bco(1), Opcode::maFusedUnary, Opcode::sLocal, 7));
}

// catch + pushloc + uinc + poploc -> in-place [immediately overwritten, no exception risk]
AFL_TEST("interpreter.Fusion:in-place:catch+pushloc+unary+poploc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maJump,  Opcode::jCatch, 1);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 4U);
    a.check("insn 1", isInstruction(bco(1), Opcode::maInplaceUnary, Opcode::sLocal, 7));
}

// pushloc + uinc + j + poploc -> in-place [overwritten in all branches]
AFL_TEST("interpreter.Fusion:in-place:catch+pushloc+jump", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maJump,  Opcode::jIfTrue, 5);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maJump,  Opcode::jAlways, 6);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 6U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maInplaceUnary, Opcode::sLocal, 7));
}

// pushloc + uinc + j + poploc -> fused, not in-place [not overwritten in all branches]
AFL_TEST("interpreter.Fusion:fused:catch+pushloc+jump", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maJump,  Opcode::jIfTrue, 4);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maJump,  Opcode::jAlways, 6);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 6U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
}

// pushloc + uinc + j + poploc -> fused, not in-place [not overwritten in all branches]
AFL_TEST("interpreter.Fusion:fused:catch+pushloc+jump:2", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maJump,  Opcode::jIfTrue, 4);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maJump,  Opcode::jAlways, 6);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 6U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
}

// pushloc + uinc + j + poploc -> fused, not in-place [infinite loop not provable]
AFL_TEST("interpreter.Fusion:fused:catch+pushloc+jump:loop", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maJump,  Opcode::jIfTrue, 2);
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 4U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
}

// pushloc + uinc + pushvar + poploc -> fused, not in-place [pushvar not provably disjoint from pushloc]
AFL_TEST("interpreter.Fusion:fused:catch+pushloc+unary+pushvar+poploc", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush,  Opcode::sLocal, 7);
    bco.addInstruction(Opcode::maUnary, interpreter::unInc, 0);
    bco.addInstruction(Opcode::maPush,  Opcode::sNamedVariable, 3);
    bco.addInstruction(Opcode::maPop,   Opcode::sLocal, 7);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 4U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedUnary, Opcode::sLocal, 7));
}


/*
 *  Test fusion with comparison.
 */

// bcmp + jcondp -> fusedcomparison
AFL_TEST("interpreter.Fusion:fused:compare+jump", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maBinary, interpreter::biCompareEQ, 0);
    bco.addInstruction(Opcode::maJump, Opcode::jIfEmpty | Opcode::jPopAlways, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sInteger, 42);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 3U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedComparison, interpreter::biCompareEQ, 0));
}

// bcmp + jcont -> not fused
AFL_TEST("interpreter.Fusion:kept:compare+jump", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maBinary, interpreter::biCompareEQ, 0);
    bco.addInstruction(Opcode::maJump, Opcode::jIfTrue, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sInteger, 42);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 3U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maBinary, interpreter::biCompareEQ, 0));
}

// pushloc + bcmp + jcond -> fusedcomparison2
AFL_TEST("interpreter.Fusion:fused:pushloc+compare+jump", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 9);
    bco.addInstruction(Opcode::maBinary, interpreter::biCompareEQ, 0);
    bco.addInstruction(Opcode::maJump, Opcode::jIfEmpty | Opcode::jPopAlways, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sInteger, 42);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 4U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maFusedComparison2, Opcode::sLocal, 9));
    a.check("insn 1", isInstruction(bco(1), Opcode::maFusedComparison, interpreter::biCompareEQ, 0));
}

/*
 *  Test miscellaneous. boundary cases.
 */

// Empty
AFL_TEST("interpreter.Fusion:empty", a)
{
    BytecodeObject bco;
    fuseInstructions(bco);
    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 0U);
}

// One
AFL_TEST("interpreter.Fusion:unit", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    fuseInstructions(bco);
    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 1U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maPush, Opcode::sLocal, 3));
}

// Fusion at place other than first
AFL_TEST("interpreter.Fusion:not-at-start", a)
{
    BytecodeObject bco;
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    bco.addInstruction(Opcode::maPush, Opcode::sLocal, 3);
    bco.addInstruction(Opcode::maBinary, interpreter::biAdd, 0);

    fuseInstructions(bco);

    a.checkEqual("getNumInstructions", bco.getNumInstructions(), 5U);
    a.check("insn 0", isInstruction(bco(0), Opcode::maPush, Opcode::sLocal, 3));
    a.check("insn 1", isInstruction(bco(1), Opcode::maPush, Opcode::sLocal, 3));
    a.check("insn 2", isInstruction(bco(2), Opcode::maPush, Opcode::sLocal, 3));
    a.check("insn 3", isInstruction(bco(3), Opcode::maFusedBinary, Opcode::sLocal, 3));
    a.check("insn 4", isInstruction(bco(4), Opcode::maBinary,      interpreter::biAdd, 0));
}
