/**
  *  \file test/gfx/gen/spaceviewtest.cpp
  *  \brief Test for gfx::gen::SpaceView
  */

#include "gfx/gen/spaceview.hpp"

#include "afl/test/testrunner.hpp"
#include "gfx/rgbapixmap.hpp"
#include "util/randomnumbergenerator.hpp"
#include <cstdio>

namespace {
    void verify(afl::test::Assert a, const gfx::RGBAPixmap& pix, afl::base::Memory<const uint32_t> expect)
    {
        afl::base::Memory<const uint32_t> got(pix.pixels());
#if 1
        a.checkEqualContent("content", got, expect);
#else
        // Generate the test code
        std::printf("static const uint32_t %s[] = {\n", "<name>");
        while (!got.empty()) {
            afl::base::Memory<const uint32_t> row = got.split(16);
            while (const uint32_t* p = row.eat()) {
                std::printf(" 0x%08X,", *p);
            }
            std::printf("\n");
        }
        std::printf("};\n");
#endif
    }
}

/** Regression test for renderStarfield(). */
AFL_TEST("gfx.gen.SpaceView:renderStarfield", a)
{
    // Render
    // At 32x32, we generate 2 stars. The seed has been chosen to generate two VISIBLE stars.
    afl::base::Ref<gfx::RGBAPixmap> pix = gfx::RGBAPixmap::create(32, 32);
    util::RandomNumberGenerator rng(1);
    gfx::gen::SpaceView(*pix).renderStarfield(rng);

    // Verify
    static const uint32_t EXPECT[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0x08080800, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0x02020200, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };
    verify(a, *pix, EXPECT);
}

/** Regression test for renderStar(). */
AFL_TEST("gfx.gen.SpaceView:renderStar", a)
{
    // Generate
    afl::base::Ref<gfx::RGBAPixmap> pix = gfx::RGBAPixmap::create(16, 16);
    gfx::gen::SpaceView(*pix).renderStar(COLORQUAD_FROM_RGBA(255, 255, 255, 0), gfx::Point(7, 9), 1);

    // Verify
    static const uint32_t EXPECT[] = {
        0x14141401, 0x16161601, 0x17171702, 0x18181802, 0x19191902, 0x1A1A1A02, 0x1B1B1B02, 0x1B1B1B02, 0x1B1B1B02, 0x1A1A1A02, 0x19191902, 0x18181802, 0x17171702, 0x16161601, 0x14141401, 0x13131301,
        0x16161601, 0x18181802, 0x19191902, 0x1B1B1B02, 0x1C1C1C03, 0x1E1E1E03, 0x1E1E1E03, 0x1F1F1F03, 0x1E1E1E03, 0x1E1E1E03, 0x1C1C1C03, 0x1B1B1B02, 0x19191902, 0x18181802, 0x16161601, 0x15151501,
        0x18181802, 0x1A1A1A02, 0x1C1C1C03, 0x1E1E1E03, 0x20202004, 0x22222204, 0x23232304, 0x23232304, 0x23232304, 0x22222204, 0x20202004, 0x1E1E1E03, 0x1C1C1C03, 0x1A1A1A02, 0x18181802, 0x16161601,
        0x1A1A1A02, 0x1D1D1D03, 0x1F1F1F03, 0x22222204, 0x25252505, 0x27272705, 0x29292906, 0x29292906, 0x29292906, 0x27272705, 0x25252505, 0x22222204, 0x1F1F1F03, 0x1D1D1D03, 0x1A1A1A02, 0x18181802,
        0x1C1C1C03, 0x1F1F1F03, 0x23232304, 0x27272705, 0x2B2B2B07, 0x2E2E2E08, 0x31313109, 0x32323209, 0x31313109, 0x2E2E2E08, 0x2B2B2B07, 0x27272705, 0x23232304, 0x1F1F1F03, 0x1C1C1C03, 0x19191902,
        0x1E1E1E03, 0x22222204, 0x27272705, 0x2C2C2C07, 0x32323209, 0x3838380C, 0x3D3D3D0E, 0x3F3F3F0F, 0x3D3D3D0E, 0x3838380C, 0x32323209, 0x2C2C2C07, 0x27272705, 0x22222204, 0x1E1E1E03, 0x1B1B1B02,
        0x20202004, 0x25252505, 0x2B2B2B07, 0x32323209, 0x3B3B3B0D, 0x46464613, 0x50505019, 0x5454541B, 0x50505019, 0x46464613, 0x3B3B3B0D, 0x32323209, 0x2B2B2B07, 0x25252505, 0x20202004, 0x1C1C1C03,
        0x22222204, 0x27272705, 0x2E2E2E08, 0x3838380C, 0x46464613, 0x5A5A5A1F, 0x71717132, 0x7F7F7F3F, 0x71717132, 0x5A5A5A1F, 0x46464613, 0x3838380C, 0x2E2E2E08, 0x27272705, 0x22222204, 0x1E1E1E03,
        0x23232304, 0x29292906, 0x31313109, 0x3D3D3D0E, 0x50505019, 0x71717132, 0xB4B4B47F, 0xFFFFFFFF, 0xB4B4B47F, 0x71717132, 0x50505019, 0x3D3D3D0E, 0x31313109, 0x29292906, 0x23232304, 0x1E1E1E03,
        0x23232304, 0x29292906, 0x32323209, 0x3F3F3F0F, 0x5454541B, 0x7F7F7F3F, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x7F7F7F3F, 0x5454541B, 0x3F3F3F0F, 0x32323209, 0x29292906, 0x23232304, 0x1F1F1F03,
        0x23232304, 0x29292906, 0x31313109, 0x3D3D3D0E, 0x50505019, 0x71717132, 0xB4B4B47F, 0xFFFFFFFF, 0xB4B4B47F, 0x71717132, 0x50505019, 0x3D3D3D0E, 0x31313109, 0x29292906, 0x23232304, 0x1E1E1E03,
        0x22222204, 0x27272705, 0x2E2E2E08, 0x3838380C, 0x46464613, 0x5A5A5A1F, 0x71717132, 0x7F7F7F3F, 0x71717132, 0x5A5A5A1F, 0x46464613, 0x3838380C, 0x2E2E2E08, 0x27272705, 0x22222204, 0x1E1E1E03,
        0x20202004, 0x25252505, 0x2B2B2B07, 0x32323209, 0x3B3B3B0D, 0x46464613, 0x50505019, 0x5454541B, 0x50505019, 0x46464613, 0x3B3B3B0D, 0x32323209, 0x2B2B2B07, 0x25252505, 0x20202004, 0x1C1C1C03,
        0x1E1E1E03, 0x22222204, 0x27272705, 0x2C2C2C07, 0x32323209, 0x3838380C, 0x3D3D3D0E, 0x3F3F3F0F, 0x3D3D3D0E, 0x3838380C, 0x32323209, 0x2C2C2C07, 0x27272705, 0x22222204, 0x1E1E1E03, 0x1B1B1B02,
        0x1C1C1C03, 0x1F1F1F03, 0x23232304, 0x27272705, 0x2B2B2B07, 0x2E2E2E08, 0x31313109, 0x32323209, 0x31313109, 0x2E2E2E08, 0x2B2B2B07, 0x27272705, 0x23232304, 0x1F1F1F03, 0x1C1C1C03, 0x19191902,
        0x1A1A1A02, 0x1D1D1D03, 0x1F1F1F03, 0x22222204, 0x25252505, 0x27272705, 0x29292906, 0x29292906, 0x29292906, 0x27272705, 0x25252505, 0x22222204, 0x1F1F1F03, 0x1D1D1D03, 0x1A1A1A02, 0x18181802,
    };
    verify(a, *pix, EXPECT);
}

/** Regression test for renderNebula(). */
AFL_TEST("gfx.gen.SpaceView:renderNebula", a)
{
    // Render
    afl::base::Ref<gfx::RGBAPixmap> pix = gfx::RGBAPixmap::create(32, 32);
    util::RandomNumberGenerator rng(1);
    gfx::gen::SpaceView(*pix).renderNebula(rng, COLORQUAD_FROM_RGBA(90, 60, 90, 0), 8, 1.1, 4);

    // Verify
    static const uint32_t EXPECT[] = {
        0x00000000, 0x03020300, 0x06040601, 0x08050802, 0x06040601, 0x07040701, 0x07040701, 0x04020400, 0x04030400, 0x04030400, 0x04030400, 0x04030400, 0x04020400, 0x02010200, 0x01000100, 0x01000100,
        0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01010100, 0x02010200, 0x03020300, 0x08050802, 0x0C080C04, 0x0D090D05, 0x140D140D, 0x1E141E1D, 0x1D131D1B,
        0x1F141F1F, 0x1C121C19, 0x18101813, 0x140D140D, 0x0F0A0F07, 0x0E090E06, 0x0B070B04, 0x04020400, 0x06040601, 0x06040601, 0x06040601, 0x05030501, 0x04030400, 0x03020300, 0x01000100, 0x01000100,
        0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x02010200, 0x03020300, 0x05030500, 0x08050802, 0x0C080C05, 0x160F1610, 0x1E141E1D, 0x20152021,
        0x20152021, 0x1D131D1B, 0x1A111A16, 0x0F0A0F07, 0x150E150F, 0x130C130B, 0x0F0A0F07, 0x05030500, 0x08050802, 0x09060903, 0x08050802, 0x07040701, 0x05030501, 0x02010200, 0x00000000, 0x01000100,
        0x01000100, 0x02010200, 0x01010100, 0x01010100, 0x01000100, 0x01010100, 0x01000100, 0x01010100, 0x02010200, 0x03020300, 0x04030400, 0x07040701, 0x0B070B04, 0x130D130C, 0x1D131D1B, 0x23172327,
        0x1B121B18, 0x1B121B17, 0x1A111A15, 0x1B121B17, 0x150E150E, 0x140D140D, 0x0B070B04, 0x06040601, 0x0B070B04, 0x0B070B04, 0x09060903, 0x08050802, 0x06040601, 0x02010200, 0x00000000, 0x01000100,
        0x01000100, 0x02010200, 0x02010200, 0x02010200, 0x03020300, 0x02010200, 0x01000100, 0x02010200, 0x02010200, 0x04030400, 0x06040601, 0x07050701, 0x0F0A0F07, 0x140D140C, 0x20152021, 0x2519252C,
        0x1C131C19, 0x1A111A16, 0x1E141E1D, 0x0F0A0F07, 0x0D080D05, 0x160E160F, 0x06040601, 0x07040701, 0x0A070A03, 0x0B070B04, 0x0C080C04, 0x09060903, 0x06040601, 0x02010200, 0x00000000, 0x01000100,
        0x01000100, 0x02010200, 0x03020300, 0x04030400, 0x04020400, 0x03020300, 0x03020300, 0x02010200, 0x03020300, 0x05030500, 0x0C080C04, 0x0F0A0F07, 0x160E160F, 0x21162123, 0x281A2832, 0x271A2730,
        0x21162123, 0x2519252C, 0x19111914, 0x0F0A0F07, 0x1F141F1E, 0x18101812, 0x0D090D05, 0x0A070A03, 0x0C080C04, 0x0D090D05, 0x0D090D05, 0x0B070B04, 0x04030400, 0x01010100, 0x00000000, 0x00000000,
        0x01000100, 0x02010200, 0x04020400, 0x04020400, 0x04020400, 0x03020300, 0x04020400, 0x02010200, 0x03020300, 0x07050701, 0x0B070B04, 0x18101813, 0x24182428, 0x281A2832, 0x291B2935, 0x2619262D,
        0x1D131D1B, 0x120C120A, 0x20152021, 0x160F1610, 0x1B121B17, 0x1D131D1B, 0x120C120A, 0x100B1008, 0x150E150F, 0x120C120B, 0x0F0A0F07, 0x03020300, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x01000100, 0x02010200, 0x02010200, 0x01010100, 0x02010200, 0x02010200, 0x04020400, 0x01010100, 0x02010200, 0x07040701, 0x09060903, 0x140D140D, 0x20152020, 0x271A2732, 0x291B2935, 0x2519252C,
        0x130C130B, 0x1F141F1E, 0x2619262E, 0x0A060A03, 0x150E150E, 0x1D131D1B, 0x1E141E1D, 0x19101913, 0x18101813, 0x03020300, 0x01000100, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000100, 0x01010100, 0x01000100, 0x02010200, 0x03020300, 0x04030400, 0x08050802, 0x100B1008, 0x1C121C19, 0x2418242A, 0x291B2935, 0x2619262D,
        0x0D090D05, 0x1F151F1F, 0x23172327, 0x06040601, 0x07040701, 0x0E090E06, 0x1C121C19, 0x1F141F1E, 0x1B121B18, 0x02010200, 0x07040701, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03020300, 0x01000100, 0x01000100, 0x02010200, 0x03020300, 0x06040601, 0x09060903, 0x0E090E06, 0x140D140D, 0x1B121B17, 0x2619262E, 0x281B2833,
        0x06040601, 0x06040601, 0x08050802, 0x0D080D05, 0x130C130B, 0x1E141E1D, 0x22162224, 0x21162122, 0x0F0A0F07, 0x03020300, 0x05030500, 0x0A060A03, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x02010200, 0x01010100, 0x01000100, 0x02010200, 0x04030400, 0x07040701, 0x09060902, 0x0D080D05, 0x120C120A, 0x160F1610, 0x24182428, 0x2418242A,
        0x08050802, 0x07040701, 0x0A070A03, 0x100B1009, 0x140D140D, 0x150E150E, 0x1A111A16, 0x0F0A0F07, 0x06040601, 0x02010200, 0x03020300, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000100, 0x01000100, 0x02010200, 0x04020400, 0x06040601, 0x08050802, 0x0A060A03, 0x0D080D05, 0x110B1109, 0x1B121B17, 0x1E141E1D, 0x21162124,
        0x09060902, 0x0E090E06, 0x0E090E06, 0x130C130B, 0x0E090E06, 0x0F0A0F07, 0x120C120A, 0x0F0A0F07, 0x07050701, 0x03020300, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x02010200, 0x06040601, 0x08050802, 0x08050802, 0x08050802, 0x0A070A03, 0x0D090D05, 0x140D140C, 0x1A111A16, 0x1D131D1B, 0x22172226,
        0x1A111A15, 0x21162124, 0x22172225, 0x1C121C19, 0x0E090E06, 0x0C080C05, 0x0F0A0F07, 0x0F0A0F07, 0x09060902, 0x04030400, 0x02010200, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x01000100, 0x04030400, 0x09060903, 0x0E090E06, 0x100B1008, 0x0F0A0F07, 0x110B1109, 0x19101913, 0x1D131D1B, 0x20152021, 0x22172225, 0x2418242A,
        0x1F141F1F, 0x1B121B17, 0x291B2935, 0x07040701, 0x08050802, 0x09060902, 0x0A070A03, 0x0C080C05, 0x0A070A03, 0x07040701, 0x03020300, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x03020300, 0x00000000, 0x01000100, 0x08050802, 0x100B1008, 0x100A1008, 0x100A1008, 0x0E090E06, 0x100B1009, 0x160E160F, 0x1A111A15, 0x1C131C1A, 0x1E141E1C, 0x1C131C1A, 0x1A111A16, 0x19101913,
        0x22172225, 0x281B2834, 0x21162122, 0x06040601, 0x0A070A03, 0x0C080C05, 0x1A111A16, 0x130C130B, 0x0A070A03, 0x06040601, 0x06040601, 0x03020300, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x07040701, 0x0D090D05, 0x120C120B, 0x130C130B, 0x140D140C, 0x110B1109, 0x120C120B, 0x150E150E, 0x18101813, 0x1A111A16, 0x1B121B17, 0x1A111A16, 0x170F1711, 0x0F0A0F07, 0x0E090E06,
        0x2519252C, 0x19111914, 0x0D080D05, 0x07040701, 0x0B070B04, 0x18101812, 0x21162123, 0x1C131C1A, 0x150E150E, 0x0B070B04, 0x06040601, 0x03020300, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x08050802, 0x0F0A0F07, 0x110B1109, 0x100B1008, 0x120C120A, 0x160E160F, 0x170F1711, 0x18101812, 0x19101914, 0x1A111A15, 0x19111914, 0x160F1610, 0x140D140D, 0x0D080D05, 0x0B070B04,
        0x271A2732, 0x18101813, 0x0F0A0F07, 0x09060902, 0x08050802, 0x19101913, 0x1E141E1C, 0x1B121B17, 0x160F1610, 0x100B1009, 0x0A060A03, 0x05030500, 0x02010200, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x06040601, 0x100B1008, 0x110B1109, 0x08050802, 0x05030500, 0x150E150E, 0x160E160F, 0x160F1610, 0x18101812, 0x19101914, 0x18101812, 0x130D130C, 0x0E090E06, 0x0A070A03, 0x08050802,
        0x3A273A6C, 0x271A2731, 0x160F1610, 0x100B1008, 0x160E160F, 0x150E150E, 0x160E160F, 0x130C130B, 0x0C080C05, 0x09060902, 0x06040601, 0x02010200, 0x01010100, 0x01000100, 0x00000000, 0x00000000,
        0x00000000, 0x02010200, 0x0E090E06, 0x0B070B04, 0x03020300, 0x03020300, 0x0E090E06, 0x150E150E, 0x140D140C, 0x170F1711, 0x18101812, 0x18101812, 0x130D130C, 0x0B070B04, 0x08050802, 0x06040601,
        0x3F2A3F7D, 0x38253865, 0x3120314B, 0x271A2732, 0x1E141E1C, 0x150E150E, 0x100B1008, 0x0E090E06, 0x08050802, 0x0A070A03, 0x02010200, 0x01010100, 0x01000100, 0x00000000, 0x00000000, 0x00000000,
        0x05030500, 0x0A070A03, 0x0B070B04, 0x06040601, 0x02010200, 0x01000100, 0x01010100, 0x06040601, 0x130C130B, 0x160E160F, 0x160F1610, 0x18101812, 0x0E090E06, 0x0A070A03, 0x08050802, 0x06040601,
        0x39263966, 0x3020304A, 0x291B2936, 0x1F151F1F, 0x1B121B17, 0x130C130B, 0x0C080C04, 0x09060902, 0x06040601, 0x05030501, 0x04020400, 0x02010200, 0x01000100, 0x02010200, 0x03020300, 0x05030501,
        0x07040701, 0x08050802, 0x0A070A03, 0x05030501, 0x02010200, 0x03020300, 0x110B1109, 0x100B1008, 0x110B1109, 0x130C130B, 0x140D140D, 0x140D140D, 0x0C080C04, 0x08050802, 0x07040701, 0x06040601,
        0x3724375F, 0x3020304A, 0x2A1C2A37, 0x1F141F1E, 0x160F1610, 0x120C120A, 0x0C080C04, 0x08050802, 0x06040601, 0x05030500, 0x04020400, 0x03020300, 0x01010100, 0x01010100, 0x02010200, 0x04030400,
        0x05030501, 0x07040701, 0x09060902, 0x08050802, 0x07040701, 0x05030500, 0x09060903, 0x07050701, 0x09060902, 0x0D090D05, 0x140D140D, 0x140D140D, 0x0D090D05, 0x08050802, 0x07050701, 0x07050701,
        0x34233458, 0x2E1F2E44, 0x24182429, 0x1D131D1B, 0x18101812, 0x100B1009, 0x0A070A03, 0x09060902, 0x07040701, 0x05030500, 0x04030400, 0x03020300, 0x02010200, 0x02010200, 0x02010200, 0x03020300,
        0x04030400, 0x05030501, 0x07040701, 0x07050701, 0x0A070A03, 0x06040601, 0x04030400, 0x06040601, 0x0C080C05, 0x130C130B, 0x140D140D, 0x140D140C, 0x0D080D05, 0x08050802, 0x08050802, 0x09060902,
        0x33223353, 0x2D1E2D40, 0x130D130C, 0x130C130B, 0x110B1109, 0x0D080D05, 0x09060902, 0x0A060A03, 0x08050802, 0x06040601, 0x05030500, 0x03020300, 0x02010200, 0x02010200, 0x02010200, 0x02010200,
        0x03020300, 0x04030400, 0x06040601, 0x07040701, 0x09060902, 0x09060902, 0x08050802, 0x08050802, 0x100A1008, 0x0F0A0F07, 0x0E090E06, 0x130C130B, 0x0E090E06, 0x09060902, 0x08050802, 0x0A060A03,
        0x3020304A, 0x0F0A0F07, 0x0B070B04, 0x09060902, 0x08050802, 0x09060902, 0x07040701, 0x09060902, 0x07040701, 0x07040701, 0x05030500, 0x03020300, 0x02010200, 0x02010200, 0x02010200, 0x02010200,
        0x02010200, 0x03020300, 0x04030400, 0x06040601, 0x07040701, 0x08050802, 0x09060903, 0x0B070B04, 0x0D080D05, 0x0A070A03, 0x08050802, 0x0D090D05, 0x0C080C04, 0x08050802, 0x08050802, 0x0A070A03,
        0x160E160F, 0x130D130C, 0x09060903, 0x06040601, 0x05030501, 0x05030500, 0x04030400, 0x06040601, 0x07040701, 0x06040601, 0x05030501, 0x03020300, 0x02010200, 0x02010200, 0x01010100, 0x01010100,
        0x02010200, 0x02010200, 0x03020300, 0x04030400, 0x05030501, 0x07040701, 0x08050802, 0x0A060A03, 0x08050802, 0x08050802, 0x06040601, 0x06040601, 0x06040601, 0x07040701, 0x08050802, 0x0B070B04,
        0x130C130B, 0x0D080D05, 0x0E090E06, 0x04030400, 0x03020300, 0x03020300, 0x03020300, 0x04030400, 0x07040701, 0x06040601, 0x06040601, 0x04030400, 0x03020300, 0x02010200, 0x01010100, 0x01010100,
        0x01010100, 0x01010100, 0x02010200, 0x03020300, 0x03020300, 0x04030400, 0x06040601, 0x07040701, 0x07040701, 0x06040601, 0x04030400, 0x04030400, 0x06040601, 0x0A060A03, 0x0E090E06, 0x120C120A,
        0x120C120B, 0x100A1008, 0x08050802, 0x09060902, 0x03020300, 0x130C130B, 0x150E150E, 0x0D080D05, 0x0C080C04, 0x09060902, 0x06040601, 0x04030400, 0x03020300, 0x02010200, 0x01010100, 0x01000100,
        0x01000100, 0x01000100, 0x01000100, 0x01010100, 0x02010200, 0x02010200, 0x03020300, 0x04030400, 0x06040601, 0x06040601, 0x04030400, 0x04020400, 0x05030500, 0x08050802, 0x0F0A0F07, 0x120C120A,
        0x120C120A, 0x0D080D05, 0x09060903, 0x04030400, 0x04020400, 0x19101914, 0x18101813, 0x120C120A, 0x0C080C04, 0x08050802, 0x05030500, 0x03020300, 0x01000100, 0x01010100, 0x01000100, 0x01000100,
        0x01000100, 0x01000100, 0x01000100, 0x01010100, 0x01000100, 0x01000100, 0x01010100, 0x02010200, 0x04020400, 0x04030400, 0x03020300, 0x03020300, 0x03020300, 0x05030501, 0x09060903, 0x100A1008,
        0x100B1009, 0x0A070A03, 0x07050701, 0x06040601, 0x06040601, 0x22172226, 0x18101812, 0x100A1008, 0x09060902, 0x06040601, 0x04030400, 0x01010100, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x01000100, 0x01000100, 0x01000100, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x01000100, 0x04030400, 0x04020400, 0x03020300, 0x02010200, 0x03020300, 0x03020300, 0x07040701, 0x08050802,
        0x0C080C04, 0x0A070A03, 0x07040701, 0x04030400, 0x05030500, 0x2518252B, 0x1A111A16, 0x130C130B, 0x07040701, 0x09060902, 0x03020300, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x02010200, 0x04020400, 0x04030400, 0x03020300, 0x02010200, 0x02010200, 0x03020300, 0x07040701, 0x07040701,
        0x100B1008, 0x150E150F, 0x100B1009, 0x130D130C, 0x1A111A16, 0x22172225, 0x1F141F1E, 0x100B1009, 0x09060902, 0x06040601, 0x02010200, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x02010200, 0x03020300, 0x04020400, 0x03020300, 0x02010200, 0x02010200, 0x02010200, 0x04030400, 0x05030500,
        0x130C130B, 0x150E150E, 0x18101813, 0x1B121B17, 0x19111914, 0x1F141F1E, 0x170F1711, 0x0C080C04, 0x0A070A03, 0x04030400, 0x01000100, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000100, 0x01010100, 0x02010200, 0x03020300, 0x02010200, 0x02010200, 0x02010200, 0x02010200, 0x05030501,
    };
    verify(a, *pix, EXPECT);
}

/** Regression test for renderSun(). */
AFL_TEST("gfx.gen.SpaceView:renderSun", a)
{
    // Render
    afl::base::Ref<gfx::RGBAPixmap> pix = gfx::RGBAPixmap::create(32, 32);
    gfx::gen::SpaceView(*pix).renderSun(COLORQUAD_FROM_RGBA(32, 128, 255, 0), gfx::Point(10, 13), 5);

    // Verify
    static const uint32_t EXPECT[] = {
        0x020B17FF, 0x030C19FF, 0x030D1BFF, 0x030E1DFF, 0x030F1FFF, 0x041020FF, 0x041122FF, 0x041123FF, 0x041224FF, 0x041225FF, 0x041225FF, 0x041225FF, 0x041224FF, 0x041123FF, 0x041122FF, 0x041020FF,
        0x030F1FFF, 0x030E1DFF, 0x030D1BFF, 0x030C19FF, 0x020B17FF, 0x020B15FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF, 0x01060CFF, 0x01050BFF, 0x01050AFF,
        0x030D1AFF, 0x030E1CFF, 0x030F1EFF, 0x041021FF, 0x041123FF, 0x041225FF, 0x051427FF, 0x051429FF, 0x05152BFF, 0x05162BFF, 0x05162CFF, 0x05162BFF, 0x05152BFF, 0x051429FF, 0x051427FF, 0x041225FF,
        0x041123FF, 0x041021FF, 0x030F1EFF, 0x030E1CFF, 0x030D1AFF, 0x030C18FF, 0x020B16FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x02080FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050AFF,
        0x030E1CFF, 0x030F1FFF, 0x041122FF, 0x041225FF, 0x051428FF, 0x05152BFF, 0x05172EFF, 0x061831FF, 0x061933FF, 0x061A34FF, 0x061A34FF, 0x061A34FF, 0x061933FF, 0x061831FF, 0x05172EFF, 0x05152BFF,
        0x051428FF, 0x041225FF, 0x041122FF, 0x030F1FFF, 0x030E1CFF, 0x030D1AFF, 0x030C18FF, 0x020B15FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF,
        0x04101FFF, 0x041123FF, 0x041326FF, 0x05152AFF, 0x05172EFF, 0x061933FF, 0x061B36FF, 0x071D3AFF, 0x071E3DFF, 0x071F3FFF, 0x08203FFF, 0x071F3FFF, 0x071E3DFF, 0x071D3AFF, 0x061B36FF, 0x061933FF,
        0x05172EFF, 0x05152AFF, 0x041326FF, 0x041123FF, 0x04101FFF, 0x030E1CFF, 0x030D1AFF, 0x020B17FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF, 0x01050BFF,
        0x041123FF, 0x041327FF, 0x05162BFF, 0x061831FF, 0x061B36FF, 0x071E3CFF, 0x082041FF, 0x082346FF, 0x09254BFF, 0x09274DFF, 0x09274EFF, 0x09274DFF, 0x09254BFF, 0x082346FF, 0x082041FF, 0x071E3CFF,
        0x061B36FF, 0x061831FF, 0x05162BFF, 0x041327FF, 0x041123FF, 0x030F1FFF, 0x030E1CFF, 0x030C19FF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF,
        0x041326FF, 0x05162BFF, 0x061931FF, 0x071C38FF, 0x08203FFF, 0x082347FF, 0x0A284FFF, 0x0A2B57FF, 0x0B2F5DFF, 0x0C3162FF, 0x0C3263FF, 0x0C3162FF, 0x0B2F5DFF, 0x0A2B57FF, 0x0A284FFF, 0x082347FF,
        0x08203FFF, 0x071C38FF, 0x061931FF, 0x05162BFF, 0x041326FF, 0x041122FF, 0x030F1EFF, 0x030D1BFF, 0x030C18FF, 0x020B16FF, 0x020A13FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF,
        0x05152AFF, 0x061831FF, 0x071C38FF, 0x082041FF, 0x09254BFF, 0x0A2B56FF, 0x0C3162FF, 0x0D376DFF, 0x0F3C78FF, 0x10407FFF, 0x104182FF, 0x10407FFF, 0x0F3C78FF, 0x0D376DFF, 0x0C3162FF, 0x0A2B56FF,
        0x09254BFF, 0x082041FF, 0x071C38FF, 0x061831FF, 0x05152AFF, 0x041225FF, 0x041021FF, 0x030E1DFF, 0x030D1AFF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x01070FFF, 0x01070EFF, 0x01060DFF,
        0x05172EFF, 0x061B36FF, 0x08203FFF, 0x09254BFF, 0x0B2C58FF, 0x0D3468FF, 0x0F3D7AFF, 0x11478DFF, 0x14509FFF, 0x1556ACFF, 0x1658B1FF, 0x1556ACFF, 0x14509FFF, 0x11478DFF, 0x0F3D7AFF, 0x0D3468FF,
        0x0B2C58FF, 0x09254BFF, 0x08203FFF, 0x061B36FF, 0x05172EFF, 0x051428FF, 0x041123FF, 0x030F1FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x020810FF, 0x01070EFF, 0x01060DFF,
        0x061933FF, 0x071E3CFF, 0x082347FF, 0x0A2B56FF, 0x0D3468FF, 0x10407FFF, 0x134E9BFF, 0x175EBBFF, 0x1B6EDBFF, 0x1E7BF5FF, 0x2080FFFF, 0x1E7BF5FF, 0x1B6EDBFF, 0x175EBBFF, 0x134E9BFF, 0x10407FFF,
        0x0D3468FF, 0x0A2B56FF, 0x082347FF, 0x071E3CFF, 0x061933FF, 0x05152BFF, 0x041225FF, 0x041020FF, 0x030E1CFF, 0x030C19FF, 0x020B16FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01060DFF,
        0x061B36FF, 0x082041FF, 0x0A284FFF, 0x0C3162FF, 0x0F3D7AFF, 0x134E9BFF, 0x1964C7FF, 0x2080FFFF, 0x9FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x9FFFFFFF, 0x2080FFFF, 0x1964C7FF, 0x134E9BFF,
        0x0F3D7AFF, 0x0C3162FF, 0x0A284FFF, 0x082041FF, 0x061B36FF, 0x05172EFF, 0x051427FF, 0x041122FF, 0x030F1EFF, 0x030D1AFF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01070DFF,
        0x071D3AFF, 0x082346FF, 0x0A2B57FF, 0x0D376DFF, 0x11478DFF, 0x175EBBFF, 0x2080FFFF, 0xE6FFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xE6FFFFFF, 0x2080FFFF, 0x175EBBFF,
        0x11478DFF, 0x0D376DFF, 0x0A2B57FF, 0x082346FF, 0x071D3AFF, 0x061831FF, 0x051429FF, 0x041123FF, 0x030F1FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x071E3DFF, 0x09254BFF, 0x0B2F5DFF, 0x0F3C78FF, 0x14509FFF, 0x1B6EDBFF, 0x9FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x9FFFFFFF, 0x1B6EDBFF,
        0x14509FFF, 0x0F3C78FF, 0x0B2F5DFF, 0x09254BFF, 0x071E3DFF, 0x061933FF, 0x05152BFF, 0x041224FF, 0x04101FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x071F3FFF, 0x09274DFF, 0x0C3162FF, 0x10407FFF, 0x1556ACFF, 0x1E7BF5FF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x1E7BF5FF,
        0x1556ACFF, 0x10407FFF, 0x0C3162FF, 0x09274DFF, 0x071F3FFF, 0x061A34FF, 0x05162BFF, 0x041225FF, 0x041020FF, 0x030E1CFF, 0x030C18FF, 0x020B15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x08203FFF, 0x09274EFF, 0x0C3263FF, 0x104182FF, 0x1658B1FF, 0x2080FFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x2080FFFF,
        0x1658B1FF, 0x104182FF, 0x0C3263FF, 0x09274EFF, 0x08203FFF, 0x061A34FF, 0x05162CFF, 0x041225FF, 0x041020FF, 0x030E1CFF, 0x030C18FF, 0x020B16FF, 0x020913FF, 0x020811FF, 0x02080FFF, 0x01070EFF,
        0x071F3FFF, 0x09274DFF, 0x0C3162FF, 0x10407FFF, 0x1556ACFF, 0x1E7BF5FF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x1E7BF5FF,
        0x1556ACFF, 0x10407FFF, 0x0C3162FF, 0x09274DFF, 0x071F3FFF, 0x061A34FF, 0x05162BFF, 0x041225FF, 0x041020FF, 0x030E1CFF, 0x030C18FF, 0x020B15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x071E3DFF, 0x09254BFF, 0x0B2F5DFF, 0x0F3C78FF, 0x14509FFF, 0x1B6EDBFF, 0x9FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x9FFFFFFF, 0x1B6EDBFF,
        0x14509FFF, 0x0F3C78FF, 0x0B2F5DFF, 0x09254BFF, 0x071E3DFF, 0x061933FF, 0x05152BFF, 0x041224FF, 0x04101FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x071D3AFF, 0x082346FF, 0x0A2B57FF, 0x0D376DFF, 0x11478DFF, 0x175EBBFF, 0x2080FFFF, 0xE6FFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xE6FFFFFF, 0x2080FFFF, 0x175EBBFF,
        0x11478DFF, 0x0D376DFF, 0x0A2B57FF, 0x082346FF, 0x071D3AFF, 0x061831FF, 0x051429FF, 0x041123FF, 0x030F1FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x01070FFF, 0x01070EFF,
        0x061B36FF, 0x082041FF, 0x0A284FFF, 0x0C3162FF, 0x0F3D7AFF, 0x134E9BFF, 0x1964C7FF, 0x2080FFFF, 0x9FFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0x9FFFFFFF, 0x2080FFFF, 0x1964C7FF, 0x134E9BFF,
        0x0F3D7AFF, 0x0C3162FF, 0x0A284FFF, 0x082041FF, 0x061B36FF, 0x05172EFF, 0x051427FF, 0x041122FF, 0x030F1EFF, 0x030D1AFF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01070DFF,
        0x061933FF, 0x071E3CFF, 0x082347FF, 0x0A2B56FF, 0x0D3468FF, 0x10407FFF, 0x134E9BFF, 0x175EBBFF, 0x1B6EDBFF, 0x1E7BF5FF, 0x2080FFFF, 0x1E7BF5FF, 0x1B6EDBFF, 0x175EBBFF, 0x134E9BFF, 0x10407FFF,
        0x0D3468FF, 0x0A2B56FF, 0x082347FF, 0x071E3CFF, 0x061933FF, 0x05152BFF, 0x041225FF, 0x041020FF, 0x030E1CFF, 0x030C19FF, 0x020B16FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01060DFF,
        0x05172EFF, 0x061B36FF, 0x08203FFF, 0x09254BFF, 0x0B2C58FF, 0x0D3468FF, 0x0F3D7AFF, 0x11478DFF, 0x14509FFF, 0x1556ACFF, 0x1658B1FF, 0x1556ACFF, 0x14509FFF, 0x11478DFF, 0x0F3D7AFF, 0x0D3468FF,
        0x0B2C58FF, 0x09254BFF, 0x08203FFF, 0x061B36FF, 0x05172EFF, 0x051428FF, 0x041123FF, 0x030F1FFF, 0x030D1BFF, 0x030C18FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x020810FF, 0x01070EFF, 0x01060DFF,
        0x05152AFF, 0x061831FF, 0x071C38FF, 0x082041FF, 0x09254BFF, 0x0A2B56FF, 0x0C3162FF, 0x0D376DFF, 0x0F3C78FF, 0x10407FFF, 0x104182FF, 0x10407FFF, 0x0F3C78FF, 0x0D376DFF, 0x0C3162FF, 0x0A2B56FF,
        0x09254BFF, 0x082041FF, 0x071C38FF, 0x061831FF, 0x05152AFF, 0x041225FF, 0x041021FF, 0x030E1DFF, 0x030D1AFF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x01070FFF, 0x01070EFF, 0x01060DFF,
        0x041326FF, 0x05162BFF, 0x061931FF, 0x071C38FF, 0x08203FFF, 0x082347FF, 0x0A284FFF, 0x0A2B57FF, 0x0B2F5DFF, 0x0C3162FF, 0x0C3263FF, 0x0C3162FF, 0x0B2F5DFF, 0x0A2B57FF, 0x0A284FFF, 0x082347FF,
        0x08203FFF, 0x071C38FF, 0x061931FF, 0x05162BFF, 0x041326FF, 0x041122FF, 0x030F1EFF, 0x030D1BFF, 0x030C18FF, 0x020B16FF, 0x020A13FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF,
        0x041123FF, 0x041327FF, 0x05162BFF, 0x061831FF, 0x061B36FF, 0x071E3CFF, 0x082041FF, 0x082346FF, 0x09254BFF, 0x09274DFF, 0x09274EFF, 0x09274DFF, 0x09254BFF, 0x082346FF, 0x082041FF, 0x071E3CFF,
        0x061B36FF, 0x061831FF, 0x05162BFF, 0x041327FF, 0x041123FF, 0x030F1FFF, 0x030E1CFF, 0x030C19FF, 0x020B17FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF,
        0x04101FFF, 0x041123FF, 0x041326FF, 0x05152AFF, 0x05172EFF, 0x061933FF, 0x061B36FF, 0x071D3AFF, 0x071E3DFF, 0x071F3FFF, 0x08203FFF, 0x071F3FFF, 0x071E3DFF, 0x071D3AFF, 0x061B36FF, 0x061933FF,
        0x05172EFF, 0x05152AFF, 0x041326FF, 0x041123FF, 0x04101FFF, 0x030E1CFF, 0x030D1AFF, 0x020B17FF, 0x020A15FF, 0x020913FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF, 0x01050BFF,
        0x030E1CFF, 0x030F1FFF, 0x041122FF, 0x041225FF, 0x051428FF, 0x05152BFF, 0x05172EFF, 0x061831FF, 0x061933FF, 0x061A34FF, 0x061A34FF, 0x061A34FF, 0x061933FF, 0x061831FF, 0x05172EFF, 0x05152BFF,
        0x051428FF, 0x041225FF, 0x041122FF, 0x030F1FFF, 0x030E1CFF, 0x030D1AFF, 0x030C18FF, 0x020B15FF, 0x020A14FF, 0x020912FF, 0x020810FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF,
        0x030D1AFF, 0x030E1CFF, 0x030F1EFF, 0x041021FF, 0x041123FF, 0x041225FF, 0x051427FF, 0x051429FF, 0x05152BFF, 0x05162BFF, 0x05162CFF, 0x05162BFF, 0x05152BFF, 0x051429FF, 0x051427FF, 0x041225FF,
        0x041123FF, 0x041021FF, 0x030F1EFF, 0x030E1CFF, 0x030D1AFF, 0x030C18FF, 0x020B16FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x02080FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050AFF,
        0x020B17FF, 0x030C19FF, 0x030D1BFF, 0x030E1DFF, 0x030F1FFF, 0x041020FF, 0x041122FF, 0x041123FF, 0x041224FF, 0x041225FF, 0x041225FF, 0x041225FF, 0x041224FF, 0x041123FF, 0x041122FF, 0x041020FF,
        0x030F1FFF, 0x030E1DFF, 0x030D1BFF, 0x030C19FF, 0x020B17FF, 0x020B15FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01060DFF, 0x01060CFF, 0x01060CFF, 0x01050BFF, 0x01050AFF,
        0x020A15FF, 0x020B17FF, 0x030C18FF, 0x030D1AFF, 0x030D1BFF, 0x030E1CFF, 0x030F1EFF, 0x030F1FFF, 0x04101FFF, 0x041020FF, 0x041020FF, 0x041020FF, 0x04101FFF, 0x030F1FFF, 0x030F1EFF, 0x030E1CFF,
        0x030D1BFF, 0x030D1AFF, 0x030C18FF, 0x020B17FF, 0x020A15FF, 0x020A14FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050AFF, 0x01050AFF,
        0x020913FF, 0x020A14FF, 0x020B16FF, 0x020B17FF, 0x030C18FF, 0x030C19FF, 0x030D1AFF, 0x030D1BFF, 0x030D1BFF, 0x030E1CFF, 0x030E1CFF, 0x030E1CFF, 0x030D1BFF, 0x030D1BFF, 0x030D1AFF, 0x030C19FF,
        0x030C18FF, 0x020B17FF, 0x020B16FF, 0x020A14FF, 0x020913FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050AFF, 0x01050AFF, 0x010409FF,
        0x020811FF, 0x020912FF, 0x020A13FF, 0x020A14FF, 0x020A15FF, 0x020B16FF, 0x020B17FF, 0x030C18FF, 0x030C18FF, 0x030C18FF, 0x030C18FF, 0x030C18FF, 0x030C18FF, 0x030C18FF, 0x020B17FF, 0x020B16FF,
        0x020A15FF, 0x020A14FF, 0x020A13FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x02080FFF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050AFF, 0x01050AFF, 0x010409FF, 0x010409FF,
        0x020810FF, 0x020811FF, 0x020912FF, 0x020912FF, 0x020913FF, 0x020A14FF, 0x020A14FF, 0x020A15FF, 0x020A15FF, 0x020B15FF, 0x020B16FF, 0x020B15FF, 0x020A15FF, 0x020A15FF, 0x020A14FF, 0x020A14FF,
        0x020913FF, 0x020912FF, 0x020912FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060DFF, 0x01060CFF, 0x01050BFF, 0x01050BFF, 0x01050AFF, 0x010409FF, 0x010409FF, 0x010408FF,
        0x01070FFF, 0x01070FFF, 0x020810FF, 0x020811FF, 0x020811FF, 0x020912FF, 0x020912FF, 0x020913FF, 0x020913FF, 0x020913FF, 0x020913FF, 0x020913FF, 0x020913FF, 0x020913FF, 0x020912FF, 0x020912FF,
        0x020811FF, 0x020811FF, 0x020810FF, 0x01070FFF, 0x01070FFF, 0x01070EFF, 0x01060DFF, 0x01060CFF, 0x01060CFF, 0x01050BFF, 0x01050AFF, 0x01050AFF, 0x010409FF, 0x010409FF, 0x010408FF, 0x010408FF,
    };
    verify(a, *pix, EXPECT);
}
