/**
  *  \file test/game/v3/loadertest.cpp
  *  \brief Test for game::v3::Loader
  */

#include "game/v3/loader.hpp"

#include "afl/charset/utf8charset.hpp"
#include "afl/except/fileformatexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/internaldirectory.hpp"
#include "afl/io/internalfilesystem.hpp"
#include "afl/io/nullfilesystem.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/test/testrunner.hpp"
#include "game/map/configuration.hpp"
#include "game/map/planet.hpp"
#include "game/map/ship.hpp"
#include "game/map/universe.hpp"
#include "game/test/interpreterinterface.hpp"
#include "game/test/registrationkey.hpp"
#include "game/test/specificationloader.hpp"
#include "game/test/stringverifier.hpp"
#include "game/v3/command.hpp"
#include "game/v3/commandcontainer.hpp"
#include "game/v3/commandextra.hpp"

using game::Element;
using game::map::Planet;
using game::map::Ship;

namespace {
    struct TestHarness {
        // Environment [Loader/Root]
        afl::base::Ref<afl::io::InternalDirectory> dir;
        afl::string::NullTranslator tx;
        afl::sys::Log log;
        afl::io::NullFileSystem fs;
        afl::charset::Utf8Charset charset;

        // Testee [Loader]
        game::v3::Loader testee;

        // Environment [loadTurnfile]
        game::Root root;
        game::Turn turn;

        TestHarness()
            : dir(afl::io::InternalDirectory::create("spec")),
              tx(), log(), fs(),
              charset(),
              testee(charset, tx, log),
              root(dir, *new game::test::SpecificationLoader(),
                   game::HostVersion(),
                   std::auto_ptr<game::RegistrationKey>(new game::test::RegistrationKey(game::RegistrationKey::Unregistered, 5)),
                   std::auto_ptr<game::StringVerifier>(new game::test::StringVerifier()),
                   std::auto_ptr<afl::charset::Charset>(new afl::charset::Utf8Charset()),
                   game::Root::Actions_t()),
              turn()
            { }
    };

    /* Player number used for generating the following turn files */
    const int PLAYER = 7;

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("vvA3", 1, 9, "abc"))' | xxd -i
       Single ShipChangeFc(9) command */
    const uint8_t SHIP_TURN[] = {
        0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x22, 0x00, 0x00, 0x00, 0x01, 0x00, 0x09, 0x00, 0x61, 0x62, 0x63, 0x4d, 0x11, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00,
        0xcb, 0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00,
        0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00,
        0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00,
        0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00,
        0xa0, 0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00,
        0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00,
        0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00,
        0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00,
        0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x4d, 0x11, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("v3", 32, 270, 5))' | xxd -i
       Single PlanetColonistTax(270) command */
    const uint8_t PLANET_TURN[] = {
        0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x22, 0x00, 0x00, 0x00, 0x20, 0x00, 0x0e, 0x01, 0x05, 0x00, 0x51, 0x10, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xcb,
        0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60,
        0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20,
        0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0,
        0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0,
        0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0,
        0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80,
        0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40,
        0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00,
        0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x51, 0x10, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a,
        0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* perl perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("v3", 52, 400, 3))' | xxd -i
       Single BaseChangeMission(400) command */
    const uint8_t BASE_TURN[] = {
        0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x22, 0x00, 0x00, 0x00, 0x34, 0x00, 0x90, 0x01, 0x03, 0x00, 0xe5, 0x10, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xcb,
        0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60,
        0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20,
        0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0,
        0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0,
        0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0,
        0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80,
        0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40,
        0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00,
        0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0xe5, 0x10, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a,
        0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("vvA3", 1, 9, "abc"), pack("v3", 32, 270, 5), pack("v3", 52, 400, 3))'  | xxd -i
       This is ShipChangeFc(9,"abc"), PlanetColonistTax(270,5), BaseChangeMission(400,3) */
    const uint8_t THREE_COMMAND_TURN[] = {
        0x07, 0x00, 0x03, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x2a, 0x00, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00, 0x01, 0x00, 0x09, 0x00, 0x61, 0x62, 0x63,
        0x20, 0x00, 0x0e, 0x01, 0x05, 0x00, 0x34, 0x00, 0x90, 0x01, 0x03, 0x00, 0xbb, 0x12, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xcb, 0x02, 0x00, 0x00,
        0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00,
        0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00,
        0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00,
        0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00,
        0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00,
        0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00,
        0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00,
        0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00,
        0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0xbb, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00
    };

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("vvA3", 1, 9, "abc"), pack("vvA3", 1, 9, "ff3"), pack("vvA3", 1, 9, "ee4"), pack("vvA3", 1, 9, "ghi"))' | xxd -i
       This is ShipChangeFc(9, "abc"), ..."ff3", "ee4", "ghi" */
    const uint8_t ALLIES_COMMAND_TURN[] = {
        0x07, 0x00, 0x04, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x2e, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00, 0x01, 0x00, 0x09,
        0x00, 0x61, 0x62, 0x63, 0x01, 0x00, 0x09, 0x00, 0x66, 0x66, 0x33, 0x01, 0x00, 0x09, 0x00, 0x65, 0x65, 0x34, 0x01, 0x00, 0x09, 0x00, 0x67, 0x68,
        0x69, 0x63, 0x15, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xcb, 0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00,
        0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00,
        0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00,
        0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00,
        0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00,
        0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00,
        0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00,
        0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x63, 0x15, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("vvvA3", 60, 3, 7, 11, "nop"))' | xxd -i
       This is SendMessage(7,11,"abc") */
    const uint8_t MESSAGE_COMMAND_TURN[] = {
        0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x22, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x03, 0x00, 0x07, 0x00, 0x0b, 0x00, 0x6e, 0x6f, 0x70, 0xbb, 0x11, 0x00, 0x00,
        0x2a, 0x00, 0x00, 0x00, 0xcb, 0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00,
        0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00,
        0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00,
        0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00,
        0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00,
        0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00,
        0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00,
        0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00,
        0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0xbb, 0x11, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* perl -I../c2systest -Mc2service -e 'print c2service::vp_make_turn(7, "22-33-4444:55:66:77", pack("v*", 100, 1,2,3))' | xxd -i
       100 is an invalid command */
    const uint8_t INVALID_COMMAND_TURN[] = {
        0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37,
        0x00, 0x00, 0xaf, 0x03, 0x78, 0x22, 0x00, 0x00, 0x00, 0x64, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00, 0x87, 0x10, 0x00, 0x00, 0x2a, 0x00, 0x00,
        0x00, 0xcb, 0x02, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00, 0x00, 0xc0, 0x09, 0x00,
        0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00, 0x80, 0x13, 0x00,
        0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00,
        0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00,
        0x00, 0xa0, 0x28, 0x00, 0x00, 0xa0, 0x01, 0x00, 0x00, 0x40, 0x03, 0x00, 0x00, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x20, 0x08, 0x00,
        0x00, 0xc0, 0x09, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00,
        0x00, 0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00,
        0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00,
        0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x07, 0x24, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x87, 0x10, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00,
        0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00
    };

    /* The timestamp used in all of the above. Yeah I know it's wrong... */
    const uint8_t MOCK_TIMESTAMP[] = {
        0x32, 0x32, 0x2d, 0x33, 0x33, 0x2d, 0x34, 0x34, 0x34, 0x34, 0x3a, 0x35, 0x35, 0x3a, 0x36, 0x36, 0x3a, 0x37
    };
}


AFL_TEST("game.v3.Loader:loadPlanets", a)
{
    // Data
    // royale:/home/home/stefan/pcc-v2/testgames/bird2/pdata4.dat
    static const uint8_t DATA[] = {
        /*0x06, 0x00,*/ 0x04, 0x00, 0x14, 0x00, 0x66, 0x30, 0x41, 0xcd, 0x00, 0x6f,
        0x00, 0x3f, 0x00, 0x20, 0x03, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0xd2,
        0x02, 0x00, 0x00, 0x6f, 0x02, 0x00, 0x00, 0xed, 0x00, 0x00, 0x00, 0x48,
        0x01, 0x00, 0x00, 0x99, 0x02, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0xa0,
        0x04, 0x00, 0x00, 0xd7, 0x0f, 0x00, 0x00, 0x32, 0x02, 0x00, 0x00, 0x62,
        0x00, 0x17, 0x00, 0x58, 0x00, 0x4e, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x64,
        0x00, 0x64, 0x00, 0x04, 0x00, 0x0f, 0x45, 0x01, 0x00, 0x07, 0x00, 0x35,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x16, 0x00, 0x61, 0x6e, 0x41, 0x02, 0x00,
        0x0f, 0x00, 0x04, 0x00, 0x68, 0x00, 0x00, 0x00, 0x96, 0x03, 0x00, 0x00,
        0x01, 0x04, 0x00, 0x00, 0x52, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x5f, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0xdf, 0x24, 0x00, 0x00,
        0xd5, 0x1e, 0x00, 0x00, 0x0b, 0x01, 0x00, 0x00, 0x9c, 0x18, 0x00, 0x00,
        0x24, 0x00, 0x44, 0x00, 0x4c, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0a, 0x00, 0x00, 0x00, 0x04, 0x00, 0x78, 0x00, 0x38, 0x41, 0x75, 0x0c,
        0x00, 0x35, 0x00, 0x02, 0x00, 0x19, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x0f, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00,
        0x00, 0x6a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xeb, 0x1c, 0x00,
        0x00, 0xf8, 0x0d, 0x00, 0x00, 0x89, 0x07, 0x00, 0x00, 0x4b, 0x07, 0x00,
        0x00, 0x56, 0x00, 0x0f, 0x00, 0x31, 0x00, 0x4b, 0x00, 0x0a, 0x00, 0x00,
        0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x3c, 0x00, 0x00, 0x00, 0x04, 0x00, 0xd9, 0x00, 0x2e, 0x41, 0x3a,
        0x82, 0x00, 0x83, 0x00, 0x2b, 0x00, 0xd8, 0x04, 0x00, 0x00, 0x9f, 0x02,
        0x00, 0x00, 0xb8, 0x04, 0x00, 0x00, 0x81, 0x06, 0x00, 0x00, 0x56, 0x04,
        0x00, 0x00, 0xf4, 0x01, 0x00, 0x00, 0xd6, 0x02, 0x00, 0x00, 0x6f, 0x00,
        0x00, 0x00, 0x37, 0x03, 0x00, 0x00, 0x72, 0x04, 0x00, 0x00, 0x74, 0x00,
        0x00, 0x00, 0x46, 0x00, 0x10, 0x00, 0x35, 0x00, 0x55, 0x00, 0x0b, 0x00,
        0x04, 0x00, 0x64, 0x00, 0x64, 0x00, 0x03, 0x00, 0x32, 0xbf, 0x00, 0x00,
        0x06, 0x00, 0x11, 0x00, 0x00, 0x00, 0x04, 0x00, 0x1c, 0x01, 0x2e, 0x32,
        0x44, 0x51, 0x00, 0x27, 0x00, 0x01, 0x00, 0x0a, 0x00, 0x00, 0x00, 0xfe,
        0x03, 0x00, 0x00, 0x8a, 0x00, 0x00, 0x00, 0xd1, 0x2b, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05,
        0x00, 0x00, 0x00, 0xf4, 0x0e, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05,
        0x00, 0x00, 0x00, 0x63, 0x00, 0x0c, 0x00, 0x58, 0x00, 0x5d, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x64, 0x00, 0x31, 0x00, 0x05, 0x00, 0x2d, 0x63, 0x00,
        0x00, 0x03, 0x00, 0x4f, 0x00, 0x00, 0x00, 0x04, 0x00, 0x38, 0x01, 0x2f,
        0x27, 0x66, 0x78, 0x00, 0x59, 0x00, 0x36, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x9b, 0x02, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0xbe, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x28, 0x00, 0x61, 0x00, 0x54, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x24, 0x22, 0x33, 0x2c,
        0x3d, 0x4e, 0x33, 0x31, 0x3e, 0x3a
    };

    // Environment
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    afl::sys::Log log;

    game::map::Universe univ;
    afl::io::ConstMemoryStream ms(DATA);

    // Testee
    game::v3::Loader testee(cs, tx, log);
    AFL_CHECK_SUCCEEDS(a("01. prepareUniverse"), testee.prepareUniverse(univ));
    AFL_CHECK_SUCCEEDS(a("02. loadPlanets"), testee.loadPlanets(univ, ms, 6, game::v3::Loader::LoadCurrent, game::PlayerSet_t(4)));

    // Verify
    // First planet is #20
    {
        const Planet* pl = univ.planets().get(20);
        a.checkNonNull("11. planet", pl);

        int owner;
        a.checkEqual("21. getOwner", pl->getOwner().get(owner), true);
        a.checkEqual("22. owner", owner, 4);

        a.checkEqual("31. getFriendlyCode",      pl->getFriendlyCode().orElse(""),                       "f0A");
        a.checkEqual("32. MineBuilding",         pl->getNumBuildings(game::MineBuilding).orElse(0),      205);
        a.checkEqual("33. FactoryBuilding",      pl->getNumBuildings(game::FactoryBuilding).orElse(0),   111);
        a.checkEqual("34. DefenseBuilding",      pl->getNumBuildings(game::DefenseBuilding).orElse(0),   63);
        a.checkEqual("35. Neutronium",           pl->getCargo(Element::Neutronium).orElse(0),            800);
        a.checkEqual("36. Tritanium",            pl->getCargo(Element::Tritanium).orElse(0),             207);
        a.checkEqual("37. Duranium",             pl->getCargo(Element::Duranium).orElse(0),              722);
        a.checkEqual("38. Molybdenum",           pl->getCargo(Element::Molybdenum).orElse(0),            623);
        a.checkEqual("39. Colonists",            pl->getCargo(Element::Colonists).orElse(0),             237);
        a.checkEqual("40. Supplies",             pl->getCargo(Element::Supplies).orElse(0),              328);
        a.checkEqual("41. Money",                pl->getCargo(Element::Money).orElse(0),                 665);
        a.checkEqual("42. ground Neutronium",    pl->getOreGround(Element::Neutronium).orElse(0),        39);
        a.checkEqual("43. ground Tritanium",     pl->getOreGround(Element::Tritanium).orElse(0),         1184);
        a.checkEqual("44. ground Duranium",      pl->getOreGround(Element::Duranium).orElse(0),          4055);
        a.checkEqual("45. ground Molybdenum",    pl->getOreGround(Element::Molybdenum).orElse(0),        562);
        a.checkEqual("46. density Neutronium",   pl->getOreDensity(Element::Neutronium).orElse(0),       98);
        a.checkEqual("47. density Tritanium",    pl->getOreDensity(Element::Tritanium).orElse(0),        23);
        a.checkEqual("48. density Duranium",     pl->getOreDensity(Element::Duranium).orElse(0),         88);
        a.checkEqual("49. density Molybdenum",   pl->getOreDensity(Element::Molybdenum).orElse(0),       78);
        a.checkEqual("50. getColonistTax",       pl->getColonistTax().orElse(0),                         11);
        a.checkEqual("51. getNativeTax",         pl->getNativeTax().orElse(0),                           4);
        a.checkEqual("52. getColonistHappiness", pl->getColonistHappiness().orElse(0),                   100);
        a.checkEqual("53. getNativeHappiness",   pl->getNativeHappiness().orElse(0),                     100);
        a.checkEqual("54. getNativeGovernment",  pl->getNativeGovernment().orElse(0),                    4);
        a.checkEqual("55. getNatives",           pl->getNatives().orElse(0),                             83215);
        a.checkEqual("56. getNativeRace",        pl->getNativeRace().orElse(0),                          7);
        a.checkEqual("57. getTemperature",       pl->getTemperature().orElse(0),                         100 - 53);
        a.checkEqual("58. isBuildingBase",       pl->isBuildingBase(),                                   false);
    }

    // Last planet is #312
    {
        const Planet* pl = univ.planets().get(312);
        a.checkNonNull("61. planet", pl);

        int owner;
        a.checkEqual("71. getOwner", pl->getOwner().get(owner), true);
        a.checkEqual("72. owner", owner, 4);

        a.checkEqual("81. getFriendlyCode",      pl->getFriendlyCode().orElse(""),                       "/'f");
        a.checkEqual("82. MineBuilding",         pl->getNumBuildings(game::MineBuilding).orElse(0),      120);
        a.checkEqual("83. FactoryBuilding",      pl->getNumBuildings(game::FactoryBuilding).orElse(0),   89);
        a.checkEqual("84. DefenseBuilding",      pl->getNumBuildings(game::DefenseBuilding).orElse(0),   54);
        // ...omitted...
        a.checkEqual("85. getColonistTax",       pl->getColonistTax().orElse(-1),                        0);
        a.checkEqual("86. getNativeTax",         pl->getNativeTax().orElse(-1),                          0);
        a.checkEqual("87. getColonistHappiness", pl->getColonistHappiness().orElse(0),                   100);
        a.checkEqual("88. getNativeHappiness",   pl->getNativeHappiness().orElse(0),                     100);
        a.checkEqual("89. getNativeGovernment",  pl->getNativeGovernment().orElse(-1),                   0);
        a.checkEqual("90. getNatives",           pl->getNatives().orElse(-1),                            0);
        a.checkEqual("91. getNativeRace",        pl->getNativeRace().orElse(-1),                         0);
        a.checkEqual("92. getTemperature",       pl->getTemperature().orElse(0),                         100 - 60);
        a.checkEqual("93. isBuildingBase",       pl->isBuildingBase(),                                   false);
    }
}

AFL_TEST("game.v3.Loader:loadShips", a)
{
    // Data
    // royale:/home/home/stefan/pcc-v2/testgames/bird2/ship10.dat
    static const uint8_t DATA[] = {
        /*0x05, 0x00,*/ 0x41, 0x00, 0x0a, 0x00, 0x76, 0x57, 0x79, 0x07, 0x00, 0xc8,
        0xff, 0xd3, 0xff, 0x53, 0x0a, 0xda, 0x09, 0x07, 0x00, 0x2e, 0x00, 0x02,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x4f,
        0x72, 0x74, 0x65, 0x67, 0x61, 0x20, 0x42, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xc5, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0xad, 0x00, 0x0a, 0x00, 0x29, 0x52, 0x50, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x7b, 0x0a, 0x38, 0x0a, 0x07, 0x00, 0x2e, 0x00, 0x02, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x44, 0x75,
        0x6e, 0x67, 0x61, 0x20, 0x42, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x8d, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00,
        0x11, 0x01, 0x0a, 0x00, 0x2f, 0x58, 0x3a, 0x07, 0x00, 0x00, 0x00, 0x00,
        0x00, 0xb7, 0x09, 0xb7, 0x09, 0x07, 0x00, 0x67, 0x00, 0x05, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x32, 0x00, 0x04, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00, 0x4d, 0x61, 0x6e,
        0x62, 0x65, 0x72, 0x72, 0x79, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xad,
        0x01, 0x0a, 0x00, 0x26, 0x57, 0x58, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x0a, 0x4a, 0x09, 0x07, 0x00, 0x57, 0x00, 0x05, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x4e, 0x65, 0x75, 0x6a,
        0x61, 0x68, 0x72, 0x73, 0x67, 0x65, 0x73, 0x63, 0x68, 0x65, 0x6e, 0x6b,
        0x20, 0x20, 0x20, 0x20, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xcf, 0x01,
        0x0a, 0x00, 0x4a, 0x4a, 0x4a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x88,
        0x09, 0x93, 0x09, 0x07, 0x00, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x4c, 0x41, 0x52, 0x47, 0x45,
        0x20, 0x44, 0x45, 0x45, 0x50, 0x20, 0x53, 0x50, 0x41, 0x43, 0x45, 0x20,
        0x46, 0x52, 0x45, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x36, 0x3d,
        0x2f, 0x48, 0x39, 0x41, 0x45, 0x3e, 0x50
    };

    // Environment
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    afl::sys::Log log;

    game::map::Universe univ;
    game::test::InterpreterInterface iface;
    afl::io::ConstMemoryStream ms(DATA);

    // Testee
    game::v3::Loader testee(cs, tx, log);
    AFL_CHECK_SUCCEEDS(a("01. prepareUniverse"), testee.prepareUniverse(univ));
    AFL_CHECK_SUCCEEDS(a("02. loadShips"), testee.loadShips(univ, ms, 5, game::v3::Loader::LoadCurrent, false, game::PlayerSet_t(10)));

    // Verify
    // First ship is #65
    {
        const Ship* sh = univ.ships().get(65);
        a.checkNonNull("11. ship", sh);

        int owner;
        a.checkEqual("21. getOwner", sh->getOwner().get(owner), true);
        a.checkEqual("22. owner", owner, 10);

        game::map::Point pt;
        a.checkEqual("31. getPosition", sh->getPosition().get(pt), true);
        a.checkEqual("32. getX", pt.getX(), 2643);
        a.checkEqual("33. getY", pt.getY(), 2522);

        a.checkEqual("41. getFriendlyCode", sh->getFriendlyCode().orElse(""),                       "vWy");
        a.checkEqual("42. getWaypointDX",   sh->getWaypointDX().orElse(0),                          -56);
        a.checkEqual("43. getWaypointDY",   sh->getWaypointDY().orElse(0),                          -45);
        a.checkEqual("44. getWarpFactor",   sh->getWarpFactor().orElse(0),                          7);
        a.checkEqual("45. getEngineType",   sh->getEngineType().orElse(0),                          7);
        a.checkEqual("46. getHull",         sh->getHull().orElse(0),                                46);
        a.checkEqual("47. getBeamType",     sh->getBeamType().orElse(0),                            2);
        a.checkEqual("48. getNumBeams",     sh->getNumBeams().orElse(0),                            4);
        a.checkEqual("49. getNumBays",      sh->getNumBays().orElse(-1),                            0);
        a.checkEqual("50. getTorpedoType",  sh->getTorpedoType().orElse(-1),                        0);
        a.checkEqual("51. getAmmo",         sh->getAmmo().orElse(-1),                               0);
        a.checkEqual("52. getNumLaunchers", sh->getNumLaunchers().orElse(-1),                       0);
        a.checkEqual("53. getMission",      sh->getMission().orElse(0),                             10);
        a.checkEqual("54. getPrimaryEnemy", sh->getPrimaryEnemy().orElse(-1),                       0);
        a.checkEqual("55. TowParameter",    sh->getMissionParameter(game::TowParameter).orElse(-1), 0);
        a.checkEqual("56. getDamage",       sh->getDamage().orElse(-1),                             0);
        a.checkEqual("57. getCrew",         sh->getCrew().orElse(0),                                102);
        a.checkEqual("58. Colonists",       sh->getCargo(Element::Colonists).orElse(-1),            0);
        a.checkEqual("59. PlainName",       sh->getName(game::PlainName, tx, iface),                "Ortega B");
        a.checkEqual("60. Neutronium",      sh->getCargo(Element::Neutronium).orElse(-1),           197);
        a.checkEqual("61. Tritanium",       sh->getCargo(Element::Tritanium).orElse(-1),            0);
        a.checkEqual("62. Duranium",        sh->getCargo(Element::Duranium).orElse(-1),             0);
        a.checkEqual("63. Molybdenum",      sh->getCargo(Element::Molybdenum).orElse(-1),           0);
        a.checkEqual("64. Supplies",        sh->getCargo(Element::Supplies).orElse(-1),             0);
    }
}

AFL_TEST("game.v3.Loader:loadBases", a)
{
    // Data
    // royale:/home/home/stefan/pcc-v2/testgames/bird2/bdata8.dat
    static const uint8_t DATA[] = {
        /*0x04, 0x00,*/ 0x91, 0x00, 0x08, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xa5, 0x00, 0x08, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x0a, 0x00,
        0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xab, 0x00, 0x08, 0x00, 0x59, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0xf8, 0x00, 0x08, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x2b, 0x25, 0x23, 0x26, 0x40, 0x2a, 0x45, 0x28, 0x25, 0x45

    };

    // Environment
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    afl::sys::Log log;

    game::map::Universe univ;
    afl::io::ConstMemoryStream ms(DATA);

    // Testee
    game::v3::Loader testee(cs, tx, log);
    AFL_CHECK_SUCCEEDS(a("01. prepareUniverse"), testee.prepareUniverse(univ));
    AFL_CHECK_SUCCEEDS(a("02. loadBases"), testee.loadBases(univ, ms, 4, game::v3::Loader::LoadCurrent, game::PlayerSet_t(8)));

    // Verify
    // Second base is #165
    {
        Planet* pl = univ.planets().get(165);
        a.checkNonNull("11. planet", pl);

        // Must pass the planet's internalCheck(), otherwise it will not give out any base properties!
        pl->setPosition(game::map::Point(1000, 2000));
        pl->addCurrentPlanetData(game::map::PlanetData(), game::PlayerSet_t(8));
        pl->internalCheck(game::map::Configuration(), game::PlayerSet_t(8), 15, tx, log);
        a.check("21. hasFullBaseData", pl->hasFullBaseData());
        a.check("22. hasBase", pl->hasBase());

        a.checkEqual("31. BaseDefenseBuilding",   pl->getNumBuildings(game::BaseDefenseBuilding).orElse(0), 200);
        a.checkEqual("32. getBaseDamage",         pl->getBaseDamage().orElse(-1),                           0);
        a.checkEqual("33. EngineTech",            pl->getBaseTechLevel(game::EngineTech).orElse(-1),        10);
        a.checkEqual("34. HullTech",              pl->getBaseTechLevel(game::HullTech).orElse(-1),          6);
        a.checkEqual("35. BeamTech",              pl->getBaseTechLevel(game::BeamTech).orElse(-1),          6);
        a.checkEqual("36. TorpedoTech",           pl->getBaseTechLevel(game::TorpedoTech).orElse(-1),       6);
        a.checkEqual("37. EngineTech",            pl->getBaseStorage(game::EngineTech, 1).orElse(-1),       0);
        a.checkEqual("38. EngineTech",            pl->getBaseStorage(game::EngineTech, 9).orElse(-1),       1);
        a.checkEqual("39. HullTech",              pl->getBaseStorage(game::HullTech, 1).orElse(-1),         0);
        a.checkEqual("40. HullTech",              pl->getBaseStorage(game::HullTech, 2).orElse(-1),         1);
        a.checkEqual("41. HullTech",              pl->getBaseStorage(game::HullTech, 6).orElse(-1),         1);
        a.checkEqual("42. HullTech",              pl->getBaseStorage(game::HullTech, 20).orElse(-1),        0);
        a.checkEqual("43. BeamTech",              pl->getBaseStorage(game::BeamTech, 1).orElse(-1),         3);
        a.checkEqual("44. BeamTech",              pl->getBaseStorage(game::BeamTech, 7).orElse(-1),         6);
        a.checkEqual("45. BeamTech",              pl->getBaseStorage(game::BeamTech, 10).orElse(-1),        0);
        a.checkEqual("46. TorpedoTech",           pl->getBaseStorage(game::TorpedoTech, 1).orElse(-1),      0);
        a.checkEqual("47. TorpedoTech",           pl->getBaseStorage(game::TorpedoTech, 10).orElse(-1),     0);
        a.checkEqual("48. fromTorpedoType",       pl->getCargo(Element::fromTorpedoType(1)).orElse(-1),     0);
        a.checkEqual("49. Fighters",              pl->getCargo(Element::Fighters).orElse(-1),               60);
        a.checkEqual("50. getBaseShipyardAction", pl->getBaseShipyardAction().orElse(-1),                   0);
        a.checkEqual("51. getBaseShipyardId",     pl->getBaseShipyardId().orElse(-1),                       0);
        a.checkEqual("52. getBaseMission",        pl->getBaseMission().orElse(-1),                          6);
    }
}

/** Test loadTurnfile(), success case.
    Prepare a universe with three objects.
    Load a turn file refering to the three objects.
    Load must succeed and update the objects. */
AFL_TEST("game.v3.Loader:loadTurnfile:success", a)
{
    // Prepare
    TestHarness h;
    {
        game::map::Ship* p = h.turn.universe().ships().create(9);
        game::map::ShipData sd;
        sd.friendlyCode = "xyz";
        sd.owner = PLAYER;
        p->addCurrentShipData(sd, game::PlayerSet_t(PLAYER));
        p->setPlayability(game::map::Object::Playable);
    }
    {
        game::map::Planet* p = h.turn.universe().planets().create(270);
        game::map::PlanetData pd;
        pd.friendlyCode = "xyz";
        pd.owner = PLAYER;
        pd.colonistTax = 12;
        p->addCurrentPlanetData(pd, game::PlayerSet_t(PLAYER));
        p->setPlayability(game::map::Object::Playable);
    }
    {
        game::map::Planet* p = h.turn.universe().planets().create(400);
        game::map::PlanetData pd;
        pd.friendlyCode = "qqq";
        pd.owner = PLAYER;
        pd.colonistTax = 12;
        game::map::BaseData bd;
        bd.mission = 1;
        p->addCurrentPlanetData(pd, game::PlayerSet_t(PLAYER));
        p->addCurrentBaseData(bd, game::PlayerSet_t(PLAYER));
        p->setPlayability(game::map::Object::Playable);
    }
    a.checkEqual("01. getFriendlyCode", h.turn.universe().ships().get(9)->getFriendlyCode().orElse(""), "xyz");
    a.checkEqual("02. getColonistTax",  h.turn.universe().planets().get(270)->getColonistTax().orElse(0), 12);
    a.checkEqual("03. getBaseMission",  h.turn.universe().planets().get(400)->getBaseMission().orElse(0), 1);
    h.turn.setTimestamp(MOCK_TIMESTAMP);

    // File to test
    afl::io::ConstMemoryStream file(THREE_COMMAND_TURN);

    // Test it
    h.testee.loadTurnfile(h.turn, h.root, file, PLAYER);

    // Verify result
    a.checkEqual("11. getFriendlyCode", h.turn.universe().ships().get(9)->getFriendlyCode().orElse(""), "abc");
    a.checkEqual("12. getColonistTax",  h.turn.universe().planets().get(270)->getColonistTax().orElse(0), 5);
    a.checkEqual("13. getBaseMission",  h.turn.universe().planets().get(400)->getBaseMission().orElse(0), 3);
}

/** Test loadTurnfile(), failure case: missing ship.
    Prepare an empty universe.
    Loading a turn refering to a ship must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:missing-ship", a)
{
    TestHarness h;
    afl::io::ConstMemoryStream file(SHIP_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: missing planet.
    Prepare an empty universe.
    Loading a turn refering to a planet must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:missing-planet", a)
{
    TestHarness h;
    afl::io::ConstMemoryStream file(PLANET_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: missing base.
    Prepare an empty universe.
    Loading a turn refering to a base must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:missing-base", a)
{
    TestHarness h;
    afl::io::ConstMemoryStream file(BASE_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: ship present but not played.
    Prepare a universe containing an unplayed ship.
    Loading a turn refering to that ship must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:unplayed-ship", a)
{
    TestHarness h;
    h.turn.universe().ships().create(9);
    afl::io::ConstMemoryStream file(SHIP_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: planet present but not played.
    Prepare a universe containing an unplayed planet.
    Loading a turn refering to that planet must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:unplayed-planet", a)
{
    TestHarness h;
    h.turn.universe().planets().create(270);
    afl::io::ConstMemoryStream file(PLANET_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: base present but not played.
    Prepare a universe containing an unplayed planet.
    Loading a turn refering to that planet as a base must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:unplayed-base", a)
{
    TestHarness h;
    h.turn.universe().planets().create(400);
    afl::io::ConstMemoryStream file(BASE_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: planet played but has no base.
    Prepare a universe containing a played planet without base.
    Loading a turn refering to that planet as a base must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:no-base", a)
{
    TestHarness h;
    {
        game::map::Planet* p = h.turn.universe().planets().create(400);
        game::map::PlanetData pd;
        pd.friendlyCode = "qqq";
        pd.owner = PLAYER;
        pd.colonistTax = 12;
        p->addCurrentPlanetData(pd, game::PlayerSet_t(PLAYER));
        p->setPlayability(game::map::Object::Playable);
    }
    afl::io::ConstMemoryStream file(BASE_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: invalid command.
    Loading a turn containing an invalid command must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:invalid-command", a)
{
    TestHarness h;
    afl::io::ConstMemoryStream file(INVALID_COMMAND_TURN);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: invalid file.
    Loading an invalid turn file (not a turn file) must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:invalid-file", a)
{
    TestHarness h;
    afl::io::ConstMemoryStream file(afl::base::Nothing);
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER), afl::except::FileFormatException);
}

/** Test loadTurnfile(), failure case: invalid player.
    Loading an turn file belonging to a different player must fail. */
AFL_TEST("game.v3.Loader:loadTurnfile:error:invalid-player", a)
{
    // Different player than turn image!
    const int PLAYER_HERE = PLAYER-1;

    // Prepare
    TestHarness h;
    {
        game::map::Ship* p = h.turn.universe().ships().create(9);
        game::map::ShipData sd;
        sd.friendlyCode = "xyz";
        sd.owner = PLAYER;
        p->addCurrentShipData(sd, game::PlayerSet_t(PLAYER_HERE));
        p->setPlayability(game::map::Object::Playable);
    }

    // File to test
    afl::io::ConstMemoryStream file(SHIP_TURN);

    // Test it
    AFL_CHECK_THROWS(a, h.testee.loadTurnfile(h.turn, h.root, file, PLAYER_HERE), afl::except::FileFormatException);
}

/** Test loadTurnfile(), success case, alliance command.
    Prepare universe with one ship.
    Loading a turn file containing multiple friendly code commands for that ship must produce a command. */
AFL_TEST("game.v3.Loader:loadTurnfile:alliance-command", a)
{
    // Prepare
    TestHarness h;
    {
        game::map::Ship* p = h.turn.universe().ships().create(9);
        game::map::ShipData sd;
        sd.friendlyCode = "xyz";
        sd.owner = PLAYER;
        p->addCurrentShipData(sd, game::PlayerSet_t(PLAYER));
        p->setPlayability(game::map::Object::Playable);
    }
    a.checkEqual("01. getFriendlyCode", h.turn.universe().ships().get(9)->getFriendlyCode().orElse(""), "xyz");
    h.turn.setTimestamp(MOCK_TIMESTAMP);

    // File to test
    afl::io::ConstMemoryStream file(ALLIES_COMMAND_TURN);

    // Test it
    h.testee.loadTurnfile(h.turn, h.root, file, PLAYER);

    // Verify result:
    // - last command wins
    a.checkEqual("11. getFriendlyCode", h.turn.universe().ships().get(9)->getFriendlyCode().orElse(""), "ghi");
    // - command message
    const game::v3::Command* cmd = game::v3::CommandExtra::create(h.turn).create(PLAYER).getCommand(game::v3::Command::TAlliance, 0);
    a.checkNonNull("12. command", cmd);
    a.checkEqual("13. getArg", cmd->getArg(), "ff3ee4");
}

/** Test loadTurnfile(), success case, message command.
    Loading a turn file containing a SendMessage command must produce an outbox message. */
AFL_TEST("game.v3.Loader:loadTurnfile:message-command", a)
{
    // Prepare
    TestHarness h;
    h.turn.setTimestamp(MOCK_TIMESTAMP);

    // File to test
    afl::io::ConstMemoryStream file(MESSAGE_COMMAND_TURN);

    // Test it
    h.testee.loadTurnfile(h.turn, h.root, file, PLAYER);

    // Verify result
    a.checkEqual("01", h.turn.outbox().getNumMessages(), 1U);
    a.checkEqual("02", h.turn.outbox().getMessageRawText(0), "abc");
}

/** Test loadConfiguration(), set game name from pconfig.src (base case). */
AFL_TEST("game.v3.Loader:game-name:pconfig", a)
{
    // Prepare
    TestHarness h;
    afl::io::InternalFileSystem fs;
    fs.createDirectory("/r");
    fs.createDirectory("/r/vpwork3");
    fs.openFile("/r/vpwork3/pconfig.src", afl::io::FileSystem::Create)
        ->fullWrite(afl::string::toBytes("gamename = testGameNamePConfig\n"));

    // Load
    h.testee.loadConfiguration(h.root, *fs.openDirectory("/r/vpwork3"));

    // Verify
    a.checkEqual("GameName", h.root.hostConfiguration()[game::config::HostConfiguration::GameName](), "testGameNamePConfig");
}

/** Test loadConfiguration(), set game name from directory name. */
AFL_TEST("game.v3.Loader:game-name:from-directory", a)
{
    // Prepare
    TestHarness h;
    afl::io::InternalFileSystem fs;
    fs.createDirectory("/r");
    fs.createDirectory("/r/vpwork3");

    // Load
    h.testee.loadConfiguration(h.root, *fs.openDirectory("/r/vpwork3"));

    // Verify
    a.checkEqual("GameName", h.root.hostConfiguration()[game::config::HostConfiguration::GameName](), "vpwork3");
}

/** Test loadConfiguration(), set game name from gamestat.dat file. */
AFL_TEST("game.v3.Loader:game-name:from-gamestat", a)
{
    // A gamestat.dat file (standard)
    static const uint8_t GAMESTAT[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x47, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x31, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x47, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x32, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x47, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x33, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x63, 0x3a, 0x5c, 0x77, 0x69, 0x6e, 0x64, 0x6f, 0x77, 0x73, 0x5c, 0x74, 0x65, 0x6d,
        0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x63, 0x3a, 0x5c, 0x77, 0x69, 0x6e, 0x64, 0x6f, 0x77, 0x73, 0x5c, 0x74, 0x65, 0x6d,
        0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x47, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x34, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x47, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x35, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x47, 0x61,
        0x6d, 0x65, 0x20, 0x20, 0x36, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3a, 0x5c, 0x73,
        0x70, 0x69, 0x65, 0x6c, 0x65, 0x5c, 0x77, 0x69, 0x6e, 0x70, 0x6c, 0x61, 0x6e, 0x5c, 0x76, 0x70,
        0x77, 0x6f, 0x72, 0x6b, 0x37, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x3a, 0x5c, 0x73,
        0x70, 0x69, 0x65, 0x6c, 0x65, 0x5c, 0x77, 0x69, 0x6e, 0x70, 0x6c, 0x61, 0x6e, 0x5c, 0x76, 0x70,
        0x77, 0x6f, 0x72, 0x6b, 0x37, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x47, 0x61, 0x6d, 0x65,
        0x20, 0x20, 0x37, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3a, 0x5c, 0x73, 0x70, 0x69,
        0x65, 0x6c, 0x65, 0x5c, 0x77, 0x69, 0x6e, 0x70, 0x6c, 0x61, 0x6e, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x47, 0x61, 0x6d, 0x65, 0x20, 0x20,
        0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00,
        0x20, 0x04, 0x00
    };

    // Prepare
    TestHarness h;

    afl::io::InternalFileSystem fs;
    fs.createDirectory("/r");
    fs.createDirectory("/r/vpwork3");
    fs.openFile("/r/gamestat.dat", afl::io::FileSystem::Create)->fullWrite(GAMESTAT);

    // Load
    h.testee.loadConfiguration(h.root, *fs.openDirectory("/r/vpwork3"));

    // Verify
    a.checkEqual("GameName", h.root.hostConfiguration()[game::config::HostConfiguration::GameName](), "Game  3");
}
