/**
  *  \file test/game/turnloadertest.cpp
  *  \brief Test for game::TurnLoader
  */

#include "game/turnloader.hpp"

#include "afl/charset/utf8charset.hpp"
#include "afl/io/filemapping.hpp"
#include "afl/io/internaldirectory.hpp"
#include "afl/io/internalfilesystem.hpp"
#include "afl/io/nullfilesystem.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/sys/internalenvironment.hpp"
#include "afl/test/testrunner.hpp"
#include "game/game.hpp"
#include "game/root.hpp"
#include "game/session.hpp"
#include "game/spec/shiplist.hpp"
#include "game/test/counter.hpp"
#include "game/test/registrationkey.hpp"
#include "game/test/specificationloader.hpp"
#include "game/test/stringverifier.hpp"
#include "game/turn.hpp"
#include "interpreter/values.hpp"
#include "util/profiledirectory.hpp"

using afl::base::Ref;
using afl::io::FileSystem;
using afl::io::InternalDirectory;
using afl::io::InternalFileSystem;
using afl::io::NullFileSystem;
using afl::string::NullTranslator;
using afl::string::Translator;
using afl::sys::InternalEnvironment;
using game::Game;
using game::HostVersion;
using game::RegistrationKey;
using game::Root;
using game::Session;
using game::Task_t;
using game::Turn;
using game::TurnLoader;

namespace {
    /* Null implementation for interface test */
    class NullTurnLoader : public TurnLoader {
     public:
        virtual PlayerStatusSet_t getPlayerStatus(int /*player*/, String_t& /*extra*/, Translator& /*tx*/) const
            { return PlayerStatusSet_t(); }
        virtual std::auto_ptr<Task_t> loadCurrentTurn(Game& /*game*/, int /*player*/, Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual std::auto_ptr<Task_t> saveCurrentTurn(const Game& /*game*/, game::PlayerSet_t /*players*/, SaveOptions_t /*opts*/, const Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual void getHistoryStatus(int /*player*/, int /*turn*/, afl::base::Memory<HistoryStatus> /*status*/, const Root& /*root*/)
            { }
        virtual std::auto_ptr<Task_t> loadHistoryTurn(Turn& /*turn*/, Game& /*game*/, int /*player*/, int /*turnNumber*/, Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual std::auto_ptr<Task_t> saveConfiguration(const Root& /*root*/, afl::sys::LogListener& /*log*/, afl::string::Translator& /*tx*/, std::auto_ptr<Task_t> then)
            { return then; }
        virtual String_t getProperty(Property /*p*/)
            { return String_t(); }
    };

    /* Null implementation that publishes the protected methods for testing */
    class PublicTurnLoader : public NullTurnLoader {
     public:
        using TurnLoader::loadCurrentDatabases;
        using TurnLoader::saveCurrentDatabases;
        using TurnLoader::defaultSaveConfiguration;
    };

    /* Common environment for testing database-related functions */
    struct DbEnvironment {
        NullTranslator tx;
        NullFileSystem fs;
        Ref<InternalDirectory> dir;
        Session session;
        Ref<Root> root;
        Ref<Game> g;

        DbEnvironment()
            : tx(), fs(), dir(InternalDirectory::create("gamedir")),
              session(tx, fs),
              root(*new Root(dir, *new game::test::SpecificationLoader(), HostVersion(),
                             std::auto_ptr<RegistrationKey>(new game::test::RegistrationKey(RegistrationKey::Registered, 10)),
                             std::auto_ptr<game::StringVerifier>(new game::test::StringVerifier()),
                             std::auto_ptr<afl::charset::Charset>(new afl::charset::Utf8Charset()),
                             Root::Actions_t())),
              g(*new Game())
            {
                for (int i = 0; i < 300; ++i) {
                    g->currentTurn().universe().planets().create(i);
                    g->currentTurn().universe().ships().create(i);
                }
            }
    };
}


/** Interface test. */
AFL_TEST_NOARG("game.TurnLoader:interface")
{
    NullTurnLoader t;
}

/** Test getDefaultPlayer(). */
AFL_TEST("game.TurnLoader:getDefaultPlayer", a)
{
    // Tester class that reports a given PlayerStatusSet_t list
    class Tester : public TurnLoader {
     public:
        Tester(afl::base::Memory<const PlayerStatusSet_t> data)
            : m_data(data)
            { }
        virtual PlayerStatusSet_t getPlayerStatus(int player, String_t& /*extra*/, Translator& /*tx*/) const
            {
                if (const PlayerStatusSet_t* p = m_data.at(player-1)) {
                    return *p;
                } else {
                    return PlayerStatusSet_t();
                }
            }
        virtual std::auto_ptr<Task_t> loadCurrentTurn(Game& /*game*/, int /*player*/, Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual std::auto_ptr<Task_t> saveCurrentTurn(const Game& /*game*/, game::PlayerSet_t /*players*/, SaveOptions_t /*opts*/, const Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual void getHistoryStatus(int /*player*/, int /*turn*/, afl::base::Memory<HistoryStatus> /*status*/, const Root& /*root*/)
            { }
        virtual std::auto_ptr<Task_t> loadHistoryTurn(Turn& /*turn*/, Game& /*game*/, int /*player*/, int /*turnNumber*/, Root& /*root*/, Session& /*session*/, std::auto_ptr<game::StatusTask_t> then)
            { return game::makeConfirmationTask(false, then); }
        virtual std::auto_ptr<Task_t> saveConfiguration(const Root& /*root*/, afl::sys::LogListener& /*log*/, afl::string::Translator& /*tx*/, std::auto_ptr<Task_t> then)
            { return then; }
        virtual String_t getProperty(Property /*p*/)
            { return String_t(); }
     private:
        afl::base::Memory<const PlayerStatusSet_t> m_data;
    };

    // Abbreviations
    const game::PlayerSet_t ALL_PLAYERS = game::PlayerSet_t::allUpTo(10);
    typedef TurnLoader::PlayerStatusSet_t PlayerStatusSet_t;

    // No player
    {
        const PlayerStatusSet_t set[] = { PlayerStatusSet_t() };
        Tester t(set);
        a.checkEqual("01. no player", t.getDefaultPlayer(ALL_PLAYERS), 0);
    }

    // Single available player
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("11. single player", t.getDefaultPlayer(ALL_PLAYERS), 3);
    }

    // Multiple available players (ambiguous)
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("21. multiple players", t.getDefaultPlayer(ALL_PLAYERS), 0);
    }

    // Multiple available players, but one is primary
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available) + TurnLoader::Primary,
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("31. primary player", t.getDefaultPlayer(ALL_PLAYERS), 4);
    }

    // Multiple available players, different order
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Available) + TurnLoader::Primary,
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("41. primary player", t.getDefaultPlayer(ALL_PLAYERS), 2);
    }

    // Multiple primaries, ambiguous
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Available) + TurnLoader::Primary,
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t(TurnLoader::Available) + TurnLoader::Primary,
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("51. multiple primaries", t.getDefaultPlayer(ALL_PLAYERS), 0);
    }

    // Primary but not available is ignored
    {
        const PlayerStatusSet_t set[] = {
            PlayerStatusSet_t(),
            PlayerStatusSet_t(TurnLoader::Primary),
            PlayerStatusSet_t(TurnLoader::Available),
            PlayerStatusSet_t()
        };
        Tester t(set);
        a.checkEqual("61. unavailable primary", t.getDefaultPlayer(ALL_PLAYERS), 3);
    }
}

/* Test loadCurrentDatabases. */
AFL_TEST("game.TurnLoader:loadCurrentDatabases", a)
{
    // Starchart DB (snippets from actual file)
    static const uint8_t CHART_DB[] = {
        // Header
        0x43, 0x43, 0x63, 0x68, 0x61, 0x72, 0x74, 0x1a, 0x01, 0x00, 0x20, 0x00, 0x01, 0x00, 0x01, 0x00, 0x07, 0x43, 0x4f, 0x4d, 0x4d, 0x45, 0x4e, 0x54,
        0x07, 0x43, 0x4f, 0x4d, 0x4d, 0x45, 0x4e, 0x54,

        // Ship property: planet 113, COMMENT = "my ship"
        0x07, 0x00, 0x11, 0x00, 0x00, 0x00, 0x71, 0x00, 0x01, 0x00, 0x00, 0x06, 0x07, 0x00, 0x00, 0x00, 0x6d, 0x79, 0x20, 0x73, 0x68, 0x69, 0x70,
    };

    // Score DB
    static const uint8_t SCORE_DB[] = {
        0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30, 0x1a, 0x9a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0x14, 0x00, 0x06, 0x00, 0x18, 0x00, 0x24, 0x00,
        0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xfc, 0xff, 0x02, 0x00, 0xe8, 0x03, 0x02, 0x00, 0x42, 0x75, 0x69, 0x6c, 0x64, 0x20, 0x50, 0x6f, 0x69, 0x6e,
        0x74, 0x73, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x50, 0x54, 0x53, 0x63, 0x6f, 0x72, 0x65, 0x20, 0x55, 0x73, 0x65, 0x64, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0xe8, 0x03, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x30, 0x33, 0x2d, 0x32, 0x39, 0x2d, 0x32, 0x30, 0x32, 0x34, 0x32, 0x33,
        0x3a, 0x30, 0x38, 0x3a, 0x30, 0x32, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x62, 0x58, 0x00, 0x00, 0x62, 0x58, 0x00, 0x00, 0x76, 0x58, 0x00, 0x00, 0x76, 0x58,
        0x00, 0x00, 0x76, 0x58, 0x00, 0x00, 0x76, 0x58, 0x00, 0x00, 0x76, 0x58, 0x00, 0x00, 0x8e, 0x59, 0x00, 0x00, 0x62, 0x58, 0x00, 0x00, 0x62, 0x58,
        0x00, 0x00, 0x62, 0x58, 0x00, 0x00
    };

    // Message configuration
    static const char*const MSG_CONFIG =
        "# PCC2 Message Configuration File\n"
        "  FILTER=(p) Planetary Message\n";

    // Teams
    static const uint8_t TEAM_CONFIG[] = {
        0x43, 0x43, 0x74, 0x65, 0x61, 0x6d, 0x30, 0x1a, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x04, 0x04,
        0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x6d, 0x79, 0x20, 0x74, 0x65, 0x61,
        0x6d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00
    };

    // Environment
    DbEnvironment env;
    env.dir->openFile("chart8.cc", FileSystem::Create)->fullWrite(CHART_DB);
    env.dir->openFile("score.cc",  FileSystem::Create)->fullWrite(SCORE_DB);
    env.dir->openFile("msg8.ini",  FileSystem::Create)->fullWrite(afl::string::toBytes(MSG_CONFIG));
    env.dir->openFile("team8.cc",  FileSystem::Create)->fullWrite(TEAM_CONFIG);

    // Test
    PublicTurnLoader testee;
    testee.loadCurrentDatabases(*env.g, 8, *env.root, env.session);

    // Verify result
    a.checkEqual("01. ship comment", interpreter::toString(env.session.world().shipProperties().get(113, interpreter::World::sp_Comment), false), "my ship");
    a.checkEqual("02. score",        env.g->scores().getFirstTurnNumber(), 1);
    a.checkEqual("03. message",      env.g->messageConfiguration().isHeadingFiltered("(p) Planetary Message"), true);
    a.checkEqual("04. team name",    env.g->teamSettings().getTeamName(8, env.tx), "my team");
}

/* Test loadCurrentDatabases, damaged files.
   This must not prevent a load. */
AFL_TEST("game.TurnLoader:loadCurrentDatabases:broken", a)
{
    // Score DB
    static const uint8_t SCORE_DB[] = {
        0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30, 0x1a, 0x9a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0x14, 0x00, 0x06, 0x00, 0x18, 0x00, 0x24, 0x00,
        0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xfc, 0xff, 0x02, 0x00, 0xe8, 0x03, 0x02, 0x00, 0x42, 0x75, 0x69, 0x6c, 0x64, 0x20, 0x50, 0x6f, 0x69, 0x6e,
        0x74, 0x73, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x50, 0x54, 0x53, 0x63, 0x6f, 0x72, 0x65, 0x20, 0x55, 0x73, 0x65, 0x64, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0xe8, 0x03, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x30, 0x33, 0x2d, 0x32, 0x39, 0x2d, 0x32, 0x30, 0x32, 0x34, 0x32, 0x33,
    };

    // Teams
    static const uint8_t TEAM_CONFIG[] = {
        0x66, 0x66, 0x66, 0x65, 0x61, 0x6d, 0x30, 0x1a, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x04, 0x04,
        0x04, 0x04, 0x04, 0x04, 0x04, 0x03, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x6d, 0x79, 0x20, 0x74, 0x65, 0x61,
    };

    // Environment
    DbEnvironment env;
    env.dir->openFile("score.cc",  FileSystem::Create)->fullWrite(SCORE_DB);
    env.dir->openFile("team8.cc",  FileSystem::Create)->fullWrite(TEAM_CONFIG);

    // Test
    PublicTurnLoader testee;
    AFL_CHECK_SUCCEEDS(a, testee.loadCurrentDatabases(*env.g, 8, *env.root, env.session));
}

/* Test loadCurrentDatabases, old stat file. */
AFL_TEST("game.TurnLoader:loadCurrentDatabases:old", a)
{
    // Score DB
    static const uint8_t SCORE_DB[] = {
        0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0x06, 0x00, 0x82, 0x00, 0x40, 0x01, 0x30, 0x36, 0x2d, 0x32, 0x32, 0x2d, 0x32, 0x30, 0x30, 0x35,
        0x32, 0x32, 0x3a, 0x34, 0x36, 0x3a, 0x34, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x0c, 0x00, 0x04, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x01,
        0x30, 0x36, 0x2d, 0x32, 0x32, 0x2d, 0x32, 0x30, 0x30, 0x35, 0x32, 0x33, 0x3a, 0x30, 0x31, 0x3a, 0x33, 0x34, 0x24, 0x00, 0x15, 0x00, 0x1d, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00,
        0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x01, 0x30, 0x39, 0x2d, 0x32, 0x34, 0x2d, 0x32, 0x30, 0x30, 0x35, 0x32, 0x33, 0x3a, 0x30,
        0x36, 0x3a, 0x34, 0x35, 0x24, 0x00, 0x15, 0x00, 0x1d, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x4a, 0x01,
        0x25, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0x01, 0x30, 0x39, 0x2d, 0x32,
        0x35, 0x2d, 0x32, 0x30, 0x30, 0x35, 0x31, 0x33, 0x3a, 0x35, 0x31, 0x3a, 0x34, 0x35, 0x24, 0x00, 0x15, 0x00, 0x1d, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x4a, 0x01, 0x25, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x67, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x48, 0x01, 0x30, 0x39, 0x2d, 0x32, 0x35, 0x2d, 0x32, 0x30, 0x30, 0x35, 0x31, 0x33, 0x3a, 0x35, 0x34, 0x3a, 0x31, 0x39,
        0x24, 0x00, 0x15, 0x00, 0x1d, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x4a, 0x01, 0x25, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x01, 0x30, 0x39, 0x2d, 0x32, 0x35, 0x2d, 0x32, 0x30,
        0x30, 0x35, 0x31, 0x33, 0x3a, 0x35, 0x37, 0x3a, 0x30, 0x32, 0x24, 0x00, 0x15, 0x00, 0x1d, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x38, 0x00, 0x4a, 0x01, 0x25, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    // Environment
    DbEnvironment env;
    env.dir->openFile("stat.cc",  FileSystem::Create)->fullWrite(SCORE_DB);

    // Test
    PublicTurnLoader testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.loadCurrentDatabases(*env.g, 8, *env.root, env.session));

    // Verify
    a.checkEqual("11. score turn", env.g->scores().getFirstTurnNumber(), 320);
    a.checkEqual("12. num scores", env.g->scores().getNumTurns(), 6U);
}

/* Test loadCurrentDatabases, empty directory. */
AFL_TEST("game.TurnLoader:loadCurrentDatabases:empty", a)
{
    // Environment
    DbEnvironment env;

    // Test
    PublicTurnLoader testee;
    AFL_CHECK_SUCCEEDS(a, testee.loadCurrentDatabases(*env.g, 8, *env.root, env.session));
}

/* Test saveCurrentDatabases. */
AFL_TEST("game.TurnLoader:saveCurrentDatabases", a)
{
    // Environment
    DbEnvironment env;
    env.session.setShipList(new game::spec::ShipList());      // Need a shiplist to write database
    env.g->teamSettings().setTeamName(7, "the team");
    env.g->messageConfiguration().setHeadingFiltered("filtered heading", true);
    env.session.world().shipProperties().create(5)->setNew(interpreter::World::sp_Comment, interpreter::makeStringValue("some comment"));

    // Test
    PublicTurnLoader testee;
    testee.saveCurrentDatabases(*env.g, 8, *env.root, env.session, env.root->charset());

    // Verify
    a.checkGreaterEqual("01. database", env.dir->openFile("chart8.cc", FileSystem::OpenRead)->getSize(), 3000U);
    a.checkDifferent("02. message", afl::string::fromBytes(env.dir->openFile("msg8.ini", FileSystem::OpenRead)->createVirtualMapping()->get()).find("filtered heading"), String_t::npos);
    a.checkDifferent("03. team",    afl::string::fromBytes(env.dir->openFile("team8.cc", FileSystem::OpenRead)->createVirtualMapping()->get()).find("the team"), String_t::npos);
}

/* Test defaultSaveConfiguration. */
AFL_TEST("game.TurnLoader:defaultSaveConfiguration", a)
{
    // Environment
    DbEnvironment env;
    env.root->userConfiguration().setOption("Backup.Turn", "some/directory/name", game::config::ConfigurationOption::Game);

    // Test
    PublicTurnLoader testee;
    game::test::Counter ctr;
    testee.defaultSaveConfiguration(*env.root, 0 /*pProfile*/, env.session.log(), env.tx, std::auto_ptr<Task_t>(Task_t::makeBound(&ctr, &game::test::Counter::increment)))
        ->call();

    // Verify
    a.checkEqual("01. did?", ctr.get(), 1);
    a.checkDifferent("02. ini", afl::string::fromBytes(env.dir->openFile("pcc2.ini", FileSystem::OpenRead)->createVirtualMapping()->get()).find("some/directory/name"), String_t::npos);
}

/* Test defaultSaveConfiguration, with profile. */
AFL_TEST("game.TurnLoader:defaultSaveConfiguration:profile", a)
{
    // Environment
    DbEnvironment env;
    env.root->userConfiguration().setOption("Backup.Turn", "some/directory/name", game::config::ConfigurationOption::User);

    // Profile
    InternalFileSystem ifs;
    InternalEnvironment ienv;
    ienv.setSettingsDirectoryName("/settings");
    util::ProfileDirectory profile(ienv, ifs);

    // Test
    PublicTurnLoader testee;
    game::test::Counter ctr;
    testee.defaultSaveConfiguration(*env.root, &profile, env.session.log(), env.tx, std::auto_ptr<Task_t>(Task_t::makeBound(&ctr, &game::test::Counter::increment)))
        ->call();

    // Verify
    a.checkEqual("01. did?", ctr.get(), 1);
    a.checkDifferent("02. ini", afl::string::fromBytes(ifs.openFile("/settings/pcc2.ini", FileSystem::OpenRead)->createVirtualMapping()->get()).find("some/directory/name"), String_t::npos);
}
