/**
  *  \file test/game/interface/vmfiletest.cpp
  *  \brief Test for game::interface::VmFile
  */

#include "game/interface/vmfile.hpp"

#include "afl/except/fileproblemexception.hpp"
#include "afl/io/filemapping.hpp"
#include "afl/io/nullfilesystem.hpp"
#include "afl/io/stream.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/test/testrunner.hpp"
#include "game/game.hpp"
#include "game/spec/shiplist.hpp"
#include "game/test/root.hpp"
#include "game/turn.hpp"
#include "interpreter/bytecodeobject.hpp"
#include "interpreter/process.hpp"
#include "interpreter/subroutinevalue.hpp"

using afl::base::Ref;
using afl::io::FileSystem;
using game::Root;
using interpreter::Process;
using interpreter::ProcessList;

namespace {
    // A file created in unit-testing environment.
    static const uint8_t SAVED_FILE[] = {
        0x43, 0x43, 0x76, 0x6D, 0x31, 0x32, 0x2D, 0x32, 0x34, 0x2D, 0x32, 0x30, 0x32, 0x31, 0x31, 0x32,
        // Sig                  Timestamp
        0x3A, 0x35, 0x30, 0x3A, 0x31, 0x34, 0x1A, 0x64, 0x04, 0x00, 0x46, 0x00, 0x07, 0x00, 0x01, 0x00,
        //                                  Sig   Vers  HeaderBytes TurnNr      PlayerNr   ||Typ:BCO
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00,
        //          ObjId 1                 ObjSize 0x58            NumProp 9              |Prop0
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                                  Prop1                                           Prop2
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        //                                  Prop3                                           Prop4
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                                  Prop5                                           Prop6
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                                  Prop7                                           Prop8
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                                  Flag        MinArgs     MaxArgs     NumLabels   'ssuspend'
        0x12, 0x0B, 0x53, 0x74, 0x6F, 0x70, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9D, 0x00,
        //          Name: "Stop"          ||Typ:Process             ObjId 0                 ObjSize 0x9D
        0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //          NumProp 7               Prop0                                           Prop1
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x01, 0x00,
        //                                  Prop2                                           Prop3
        0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                                  Prop4                                           Prop5
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00,
        //                                  Prop6                                           Pri   pkDefault
        0x01, 0x00, 0x53, 0x75, 0x73, 0x70, 0x65, 0x6E, 0x64, 0x65, 0x64, 0x20, 0x50, 0x72, 0x6F, 0x63,
        // ctos=1   Name=Suspended Process
        0x65, 0x73, 0x73, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04,
        //               ||Typ:Frame              ObjId 0                 ObjSize 0x34            Num
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
        // Prop 4        |Prop0                                           Prop1
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                Prop2                                           Prop3
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        //                BCO-ID                  pc                      contextSP               exSP
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x00,
        //                Flags                 || Tag_Global                         Tag_Frame
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9A, 0x00, 0x00, 0x00, 0x07,
        //               ||Typ:Process            ObjId 0                 ObjSize 0x9A            Num
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04,
        // Prop7          Prop0                                           Prop1
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0E, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x44,
        //                Prop2                                           Prop3
        0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                Prop4                                           Prop5
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x33, 0x00, 0x01, 0x00, 0x46,
        //                Prop6                                           Pri pkDefault ctos     |Name
        0x72, 0x6F, 0x7A, 0x65, 0x6E, 0x20, 0x50, 0x72, 0x6F, 0x63, 0x65, 0x73, 0x73, 0x03, 0x00, 0x00,
        // Frozen Process                                                           ||Typ:Frame
        0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //    ObjId 0                 ObjSize 0x34            NumProp 4             | Prop0
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //                            Prop1                                           Prop2
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        //                            Prop3                                           BCO-ID
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        //    pc                      contextSP               exceptionSP             Flags
        0x00, 0x00, 0x8A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x00,
        //    Tag_Global                          Tag_Frame
    };

    // A file created by running PCC2
    static const uint8_t REAL_FILE[] = {
        0x43, 0x43, 0x76, 0x6d, 0x30, 0x33, 0x2d, 0x32, 0x34, 0x2d, 0x32, 0x30, 0x32, 0x32, 0x30, 0x33,
        0x3a, 0x30, 0x30, 0x3a, 0x30, 0x32, 0x1a, 0x64, 0x04, 0x00, 0x27, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x12, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x99, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x32, 0x00, 0x01, 0x00, 0x43, 0x6f, 0x6e, 0x73, 0x6f, 0x6c, 0x65, 0x3a, 0x20, 0x73,
        0x74, 0x6f, 0x70, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x00,
        0x00, 0x00, 0x00
    };

    // A file containing a real auto-task for ship #113.
    static const uint8_t TASK_FILE[] = {
        0x43, 0x43, 0x76, 0x6d, 0x30, 0x33, 0x2d, 0x32, 0x39, 0x2d, 0x32, 0x30, 0x32, 0x34, 0x32, 0x33,
        0x3a, 0x30, 0x38, 0x3a, 0x30, 0x32, 0x1a, 0x64, 0x04, 0x00, 0x01, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06,
        0x04, 0x00, 0x00, 0x00, 0x73, 0x74, 0x6f, 0x70, 0x0b, 0x43, 0x43, 0x24, 0x41, 0x55, 0x54, 0x4f,
        0x45, 0x58, 0x45, 0x43, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x05,
        0x41, 0x75, 0x74, 0x6f, 0x20, 0x54, 0x61, 0x73, 0x6b, 0x20, 0x53, 0x68, 0x69, 0x70, 0x20, 0x31,
        0x31, 0x33, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3c, 0x01, 0x00, 0x00, 0x09, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x1c, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00,
        0x06, 0x00, 0x00, 0x06, 0x25, 0x00, 0x00, 0x00, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x75,
        0x72, 0x69, 0x6e, 0x67, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f,
        0x66, 0x20, 0x61, 0x75, 0x74, 0x6f, 0x20, 0x74, 0x61, 0x73, 0x6b, 0x3a, 0x20, 0x0a, 0x53, 0x59,
        0x53, 0x54, 0x45, 0x4d, 0x2e, 0x45, 0x52, 0x52, 0x09, 0x54, 0x52, 0x41, 0x4e, 0x53, 0x4c, 0x41,
        0x54, 0x45, 0x06, 0x4e, 0x4f, 0x54, 0x49, 0x46, 0x59, 0x04, 0x00, 0x10, 0x04, 0x00, 0x00, 0x01,
        0x00, 0x01, 0x00, 0x07, 0x0b, 0x00, 0x00, 0x01, 0x0b, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x05,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x09, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03,
        0x01, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x05, 0x00, 0x00, 0x07, 0x04, 0x03, 0x43, 0x4d,
        0x44, 0x43, 0x43, 0x24, 0x41, 0x55, 0x54, 0x4f, 0x45, 0x58, 0x45, 0x43, 0x2f, 0x68, 0x6f, 0x6d,
        0x65, 0x2f, 0x73, 0x74, 0x65, 0x66, 0x61, 0x6e, 0x2f, 0x73, 0x72, 0x63, 0x2f, 0x63, 0x32, 0x6e,
        0x67, 0x2f, 0x2e, 0x62, 0x75, 0x69, 0x6c, 0x64, 0x2f, 0x73, 0x68, 0x61, 0x72, 0x65, 0x2f, 0x72,
        0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2f, 0x63, 0x6f, 0x72, 0x65, 0x2e, 0x71, 0x00, 0x00,
        0x00, 0x00, 0xf9, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xfa, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0xfb, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x00, 0x00, 0xfd, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x12, 0x0b, 0x45, 0x76, 0x61, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xda, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x01, 0x02, 0x00, 0x41, 0x75, 0x74, 0x6f, 0x20, 0x54,
        0x61, 0x73, 0x6b, 0x20, 0x53, 0x68, 0x69, 0x70, 0x20, 0x31, 0x31, 0x33, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04, 0x00, 0x00, 0x00, 0x73, 0x74, 0x6f, 0x70, 0x03, 0x43,
        0x4d, 0x44, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x71, 0x00,
        0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x01, 0x00, 0x00, 0x00, 0x00, 0x89,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00
    };

    struct Environment {
        afl::io::NullFileSystem fs;
        afl::string::NullTranslator tx;
        game::Session session;

        Environment()
            : fs(), tx(), session(tx, fs)
            { }
    };

    void addGame(Environment& env)
    {
        afl::base::Ptr<game::Game> g = new game::Game();
        g->currentTurn().setTimestamp(game::Timestamp(2021, 12, 24, 12, 50, 14));
        g->currentTurn().setTurnNumber(70);
        for (int i = 1; i <= 200; ++i) {
            g->currentTurn().universe().ships().create(i);
            g->currentTurn().universe().planets().create(i);
        }
        env.session.setGame(g);
    }

    Ref<Root> addRoot(Environment& env)
    {
        Ref<Root> root = game::test::makeRoot(game::HostVersion());
        env.session.setRoot(root.asPtr());
        return root;
    }

    void addShipList(Environment& env)
    {
        env.session.setShipList(new game::spec::ShipList());
    }

    void loadTaskCommon(Environment& env, afl::test::Assert a)
    {
        Ref<Root> root = addRoot(env);
        root->gameDirectory().openFile("script8.cc", FileSystem::Create)
            ->fullWrite(TASK_FILE);
        addGame(env);
        addShipList(env);

        // Load
        game::interface::loadVM(env.session, 8);

        // Verify
        a.checkEqual("01. process list size", env.session.processList().getProcessList().size(), 1U);

        Process* p1 = env.session.processList().getProcessList()[0];
        a.checkNonNull("11. process", p1);
        a.checkEqual("12. process name", p1->getName(), "Auto Task Ship 113");
    }

    void addTerminateHandler(Environment& env)
    {
        interpreter::BCORef_t bco = interpreter::BytecodeObject::create(true);
        bco->addInstruction(interpreter::Opcode::maSpecial, interpreter::Opcode::miSpecialTerminate, 0);
        env.session.world().setNewGlobalValue("CC$AUTOTERMINATE", new interpreter::SubroutineValue(bco));
    }

    void runProcesses(Environment& env)
    {
        ProcessList& pl = env.session.processList();
        uint32_t pgid = pl.allocateProcessGroup();
        pl.resumeSuspendedProcesses(pgid);
        pl.startProcessGroup(pgid);
        pl.run(0);
    }
}

/** Test loadVM() with a synthetic file. */
AFL_TEST("game.interface.VmFile:load:synthetic", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    root->gameDirectory().openFile("script7.cc", FileSystem::Create)
        ->fullWrite(SAVED_FILE);
    addGame(env);

    // Load
    game::interface::loadVM(env.session, 7);

    // Verify
    a.checkEqual("01. process list size", env.session.processList().getProcessList().size(), 2U);

    Process* p1 = env.session.processList().getProcessList()[0];
    a.checkNonNull("11. first process", p1);
    a.checkEqual("12. first name", p1->getName(), "Suspended Process");

    Process* p2 = env.session.processList().getProcessList()[1];
    a.checkNonNull("21. second process", p2);
    a.checkEqual("22. second name", p2->getName(), "Frozen Process");
}

/** Test loadVM() with a file created by PCC2. */
AFL_TEST("game.interface.VmFile:load:real", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    root->gameDirectory().openFile("script2.cc", FileSystem::Create)
        ->fullWrite(REAL_FILE);
    addGame(env);

    // Load
    game::interface::loadVM(env.session, 2);

    // Verify
    a.checkEqual("01. process list size", env.session.processList().getProcessList().size(), 1U);

    Process* p1 = env.session.processList().getProcessList()[0];
    a.checkNonNull("11. process", p1);
    a.checkEqual("12. process name", p1->getName(), "Console: stop");

    // terminateUnusableAutoTasks will not touch this process
    size_t n = p1->getNumActiveFrames();
    game::interface::terminateUnusableAutoTasks(env.session);
    a.checkEqual("21. frames", p1->getNumActiveFrames(), n);
}

/** Test loadVM() when file does not exist. */
AFL_TEST("game.interface.VmFile:load:missing-file", a)
{
    // Environment
    Environment env;
    addRoot(env);
    addGame(env);

    // Load
    AFL_CHECK_SUCCEEDS(a("01. loadVM"), game::interface::loadVM(env.session, 2));

    // Verify
    a.checkEqual("11. process list size", env.session.processList().getProcessList().size(), 0U);
}

/** Test loadVM() with a bad file. */
AFL_TEST("game.interface.VmFile:error:bad-file", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    root->gameDirectory().openFile("script2.cc", FileSystem::Create);
    addGame(env);

    // Load
    AFL_CHECK_THROWS(a, game::interface::loadVM(env.session, 2), afl::except::FileProblemException);
}

/** Test loadVM() with a foreign file (wrong player). */
AFL_TEST("game.interface.VmFile:error:foreign", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    root->gameDirectory().openFile("script3.cc", FileSystem::Create)
        ->fullWrite(REAL_FILE);
    addGame(env);

    // Load
    AFL_CHECK_THROWS(a, game::interface::loadVM(env.session, 3), afl::except::FileProblemException);
}

/** Test saveVM(). */
AFL_TEST("game.interface.VmFile:save:full", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    addGame(env);

    // Add some processes
    interpreter::BCORef_t bco = interpreter::BytecodeObject::create(true);
    bco->setSubroutineName("Stop");
    bco->addInstruction(interpreter::Opcode::maSpecial, interpreter::Opcode::miSpecialSuspend, 0);

    // - suspended process (will be saved)
    Process& suspendedProcess = env.session.processList().create(env.session.world(), "Suspended Process");
    suspendedProcess.pushFrame(bco, false);
    suspendedProcess.setState(Process::Suspended);

    // - frozen process (will be saved)
    Process& frozenProcess = env.session.processList().create(env.session.world(), "Frozen Process");
    frozenProcess.pushFrame(bco, false);
    frozenProcess.setState(Process::Frozen);
    frozenProcess.setPriority(51);

    // - running process (will not be saved)
    Process& runningProcess = env.session.processList().create(env.session.world(), "Running Process");
    runningProcess.pushFrame(bco, false);
    runningProcess.setState(Process::Running);

    // - runnable process (will not be saved)
    Process& runnableProcess = env.session.processList().create(env.session.world(), "Runnable Process");
    runnableProcess.pushFrame(bco, false);
    runnableProcess.setState(Process::Runnable);

    // - terminated process (will not be saved)
    Process& terminatedProcess = env.session.processList().create(env.session.world(), "Terminated Process");
    terminatedProcess.pushFrame(bco, false);
    terminatedProcess.setState(Process::Terminated);

    // - failed process (will not be saved)
    Process& failedProcess = env.session.processList().create(env.session.world(), "Failed Process");
    failedProcess.pushFrame(bco, false);
    failedProcess.setState(Process::Failed);

    // Save
    game::interface::saveVM(env.session, 7);

    // Verify
    Ref<afl::io::FileMapping> m = root->gameDirectory().openFile("script7.cc", FileSystem::OpenRead)->createVirtualMapping();
    a.checkEqual("01. file size", m->get().size(), sizeof(SAVED_FILE));
    a.checkEqualContent("02. file content", m->get(), afl::base::ConstBytes_t(SAVED_FILE));
}

/** Test saveVM() with no processes.
    VM file will be deleted. */
AFL_TEST("game.interface.VmFile:save:empty", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);
    root->gameDirectory().openFile("script7.cc", FileSystem::Create);
    addGame(env);

    // Save
    game::interface::saveVM(env.session, 7);

    // Verify
    a.checkNull("01. no file created", root->gameDirectory().openFileNT("script7.cc", FileSystem::OpenRead).get());
}

/** Test saveVM() with missing preconditions. */
AFL_TEST("game.interface.VmFile:save:missing-preconditions", a)
{
    // Environment
    Environment env;
    Ref<Root> root = addRoot(env);

    // Save: no-op, because no game
    AFL_CHECK_SUCCEEDS(a, game::interface::saveVM(env.session, 7));

    // Verify
    a.checkNull("01. no file created", root->gameDirectory().openFileNT("script7.cc", FileSystem::OpenRead).get());
}

/** Test loading/terminating an auto task, termination case. */
AFL_TEST("game.interface.VmFile:terminateUnusableAutoTasks:terminate", a)
{
    // Environment
    Environment env;
    loadTaskCommon(env, a);
    addTerminateHandler(env);

    // Action
    game::interface::terminateUnusableAutoTasks(env.session);
    runProcesses(env);

    // Because ship 113 does not exist, task will be terminated by injecting CC$AUTOTERMINATE.
    // This will set process status to Terminated.
    a.checkEqual("12. process name", env.session.processList().getProcessList()[0]->getState(), Process::Terminated);
}

/** Test loading/terminating an auto task, keep case. */
AFL_TEST("game.interface.VmFile:terminateUnusableAutoTasks:keep", a)
{
    // Environment
    Environment env;
    loadTaskCommon(env, a);
    addTerminateHandler(env);

    // Add ship
    game::map::Ship& sh = *env.session.getGame()->currentTurn().universe().ships().get(113);
    sh.setOwner(10);
    sh.setPlayability(game::map::Object::Playable);

    // Action
    game::interface::terminateUnusableAutoTasks(env.session);
    runProcesses(env);

    // Task will execute normally.
    // Task was executing a STOP command which now finishes, causing the task to fall off its end (=Ended).
    a.checkEqual("12. process name", env.session.processList().getProcessList()[0]->getState(), Process::Ended);
}
