/**
  *  \file test/afl/net/http/cookiejartest.cpp
  *  \brief Test for afl::net::http::CookieJar
  */

#include "afl/net/http/cookiejar.hpp"

#include "afl/io/constmemorystream.hpp"
#include "afl/test/testrunner.hpp"

namespace {
    class Consumer : public afl::net::HeaderConsumer {
     public:
        void handleHeader(String_t key, String_t value)
            {
                if (!m_buffer.empty()) {
                    m_buffer += "||";
                }
                m_buffer += key;
                m_buffer += "==";
                m_buffer += value;
            }

        String_t m_buffer;
    };
}

/** Simple tests. */
AFL_TEST("afl.net.http.CookieJar", a)
{
    // Load a cookies.txt file
    afl::net::http::CookieJar jar;

    static const char data[] =
        "# Do not edit this file\n\n"
        "www.example.com\tFALSE\t/admin\tTRUE\t100000000000\tsession\t12345\n"
        "example.com\tTRUE\t/\tFALSE\t100000000000\tlanguage\ten\n"
        "example.com\tTRUE\t/help/\tFALSE\t100000000000\tsize\tbig\n"
        "example.com\tTRUE\t/\tFALSE\t100000000000\tcolor\tred\n";
    afl::io::ConstMemoryStream file(afl::string::toBytes(data));
    jar.loadCookies(file);

    // Tests
    {
        afl::net::Url url;
        a.check("01. parse", url.parse("http://example.com/contact.html"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("01. enumerateCookies", c.m_buffer, "language==en||color==red");
    }
    {
        afl::net::Url url;
        a.check("02. parse", url.parse("http://example.com/admin/edit"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("02. enumerateCookies", c.m_buffer, "language==en||color==red");
    }
    {
        afl::net::Url url;
        a.check("03. parse", url.parse("https://example.com/admin/edit"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("03. enumerateCookies", c.m_buffer, "language==en||color==red");
    }
    {
        afl::net::Url url;
        a.check("04. parse", url.parse("https://www.example.com/admin/edit"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("04. enumerateCookies", c.m_buffer, "session==12345||language==en||color==red");
    }
    {
        afl::net::Url url;
        a.check("05. parse", url.parse("https://www.example.com/help/basics/"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("05. enumerateCookies", c.m_buffer, "size==big||language==en||color==red");
    }

    // Modify it (precondition for next subtest)
    {
        afl::net::Url url;
        a.check("11. parse", url.parse("https://www.example.com/admin/logout"));

        jar.removeCookie(afl::net::http::Cookie(url,
                                                afl::sys::Time::getCurrentTime(),
                                                afl::net::HeaderField("session", "; path=/admin; domain=www.example.com")));
    }
    {
        afl::net::Url url;
        a.check("12. parse", url.parse("https://www.example.com/admin/edit"));

        Consumer c;
        jar.enumerateCookies(url, c);
        a.checkEqual("12. enumerateCookies", c.m_buffer, "language==en||color==red");
    }
}
